<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\NewsChatbotController;
use App\Http\Controllers\DeepDiveController;
use App\Http\Controllers\AdminController;
use App\Http\Controllers\AdController;
use App\Http\Controllers\Auth\LoginController;
use App\Http\Controllers\Auth\RegisterController;
use App\Http\Controllers\UserController;

// Authentication routes
Route::get('/login', [LoginController::class, 'showLoginForm'])->name('login');
Route::post('/login', [LoginController::class, 'login']);
Route::post('/logout', [LoginController::class, 'logout'])->name('logout');
Route::get('/register', [RegisterController::class, 'showRegistrationForm'])->name('register');
Route::post('/register', [RegisterController::class, 'register']);

// Home page - Chatbot
Route::get('/', [NewsChatbotController::class, 'index'])->name('home');

// Election Intelligence Dashboard (public for testing; re-enable 'auth' when ready)
Route::prefix('elections')->name('elections.')->group(function () {
    Route::get('/', [\App\Http\Controllers\ElectionController::class, 'dashboard'])->name('dashboard');
    Route::get('/region/{regionCode}', [\App\Http\Controllers\ElectionController::class, 'region'])->name('region');
    Route::get('/compare', [\App\Http\Controllers\ElectionController::class, 'compare'])->name('compare');
    Route::get('/polls', [\App\Http\Controllers\ElectionController::class, 'polls'])->name('polls');
    Route::post('/polls', [\App\Http\Controllers\ElectionController::class, 'storePoll'])->name('polls.store');
    Route::post('/chat', [\App\Http\Controllers\ElectionController::class, 'chat'])->name('chat');
    Route::post('/predict', [\App\Http\Controllers\ElectionController::class, 'generatePrediction'])->name('predict');
    Route::get('/api/election/{electionId}', [\App\Http\Controllers\ElectionController::class, 'apiGetElectionData'])->name('api.election');
});

// PWA Install page
Route::get('/install', function () {
    return view('pwa-install');
})->name('pwa.install');

// Dashboard (legacy)
Route::get('/dashboard', [HomeController::class, 'dashboard'])->name('dashboard');

// Chat session save/restore (FAB - requires auth)
Route::middleware('auth')->group(function () {
    Route::post('/api/chat/sessions', [NewsChatbotController::class, 'saveChatSession'])->name('api.chat.sessions.save');
    Route::get('/api/chat/sessions', [NewsChatbotController::class, 'listChatSessions'])->name('api.chat.sessions.list');
    Route::get('/api/chat/sessions/{id}', [NewsChatbotController::class, 'getChatSession'])->name('api.chat.sessions.get');
});

// News Chatbot Routes
Route::prefix('news')->group(function () {
    Route::controller(NewsChatbotController::class)->group(function () {
        Route::get('/chatbot', 'index')->name('news.chatbot');
        Route::post('/chat', 'chat')->name('news.chat');
    });
    
    // Deep Dive Routes (separate from NewsChatbotController)
    Route::get('/deep-dive/{article_id}', [DeepDiveController::class, 'show'])->name('news.deep-dive.show');
    Route::post('/deep-dive/{article_id}/chat', [DeepDiveController::class, 'chat'])->name('news.deep-dive.chat');
});

// User routes (authenticated)
Route::middleware('auth')->group(function () {
    Route::get('/user/dashboard', [UserController::class, 'dashboard'])->name('user.dashboard');
    Route::get('/user/profile/edit', [UserController::class, 'editProfile'])->name('user.profile.edit');
    Route::post('/user/profile/update', [UserController::class, 'updateProfile'])->name('user.profile.update');
    Route::get('/user/preferences/edit', [UserController::class, 'editPreferences'])->name('user.preferences.edit');
    Route::post('/user/preferences/update', [UserController::class, 'updatePreferences'])->name('user.preferences.update');
    Route::get('/user/reading-history', [UserController::class, 'readingHistory'])->name('user.reading-history');
    Route::get('/user/bookmarks', [UserController::class, 'bookmarks'])->name('user.bookmarks');
    
    // Bookmark API routes
    Route::post('/user/bookmark/add', [UserController::class, 'addBookmark'])->name('user.bookmark.add');
    Route::post('/user/bookmark/remove', [UserController::class, 'removeBookmark'])->name('user.bookmark.remove');
});


// Admin Routes
Route::prefix('admin')->name('admin.')->group(function () {
    // Login routes (no auth required)
    Route::get('/login', [AdminController::class, 'showLogin'])->name('login');
    Route::post('/login', [AdminController::class, 'login'])->name('login.post');
    
    // Protected routes (require auth) - temporarily disabled for debugging
    Route::group([], function () {
        Route::get('/', [AdminController::class, 'dashboard'])->name('dashboard');
        
        // Ad management routes
        Route::get('/ads', [AdController::class, 'index'])->name('ads');
        Route::get('/ads/create', [AdController::class, 'create'])->name('ads.create');
        Route::post('/ads', function(\Illuminate\Http\Request $request) {
            \Log::info('Ad creation request received', $request->all());
            
            // Simple validation
            $request->validate([
                'title' => 'required|string|max:255',
                'content' => 'required|string',
                'label' => 'required|string|max:100',
                'cta_text' => 'required|string|max:100',
                'cta_url' => 'required|url',
                'media_type' => 'required|in:text,image,video',
                'media_file' => 'nullable|file|mimes:jpg,jpeg,png,gif,mp4,avi,mov|max:10240',
                'media_alt' => 'nullable|string|max:255',
                'display_duration' => 'required|integer|min:0|max:10000',
                'rotation_weight' => 'required|integer|min:1|max:10',
                'is_active' => 'nullable|in:on,1,true',
                'start_date' => 'nullable|date',
                'end_date' => 'nullable|date|after:start_date',
                'max_impressions' => 'nullable|integer|min:1',
            ]);

            $data = $request->except('media_file', '_token');
            $data['is_active'] = $request->has('is_active') ? 1 : 0;

            // Handle media upload
            if ($request->hasFile('media_file')) {
                $file = $request->file('media_file');
                $filename = time() . '_' . $file->getClientOriginalName();
                $file->move(public_path('assets/images/ads'), $filename);
                $data['media_url'] = asset('assets/images/ads/' . $filename);
            }

            // Handle targeting rules
            if ($request->has('targeting_rules')) {
                $data['targeting_rules'] = json_encode($request->targeting_rules);
            }

            // Add timestamps
            $data['created_at'] = now();
            $data['updated_at'] = now();

            $id = \DB::table('ai_ads')->insertGetId($data);
            \Log::info('Ad created successfully', ['id' => $id]);

            return redirect()->route('admin.ads')->with('success', 'Ad created successfully!');
        })->name('ads.store');
        Route::get('/ads/{id}/edit', [AdController::class, 'edit'])->name('ads.edit');
        Route::put('/ads/{id}', function(\Illuminate\Http\Request $request, $id) {
            \Log::info('Ad update request received', ['id' => $id, 'data' => $request->all()]);
            
            // Simple validation
            $request->validate([
                'title' => 'required|string|max:255',
                'content' => 'required|string',
                'label' => 'required|string|max:100',
                'cta_text' => 'required|string|max:100',
                'cta_url' => 'required|url',
                'media_type' => 'required|in:text,image,video',
                'media_file' => 'nullable|file|mimes:jpg,jpeg,png,gif,mp4,avi,mov|max:10240',
                'media_alt' => 'nullable|string|max:255',
                'display_duration' => 'required|integer|min:0|max:10000',
                'rotation_weight' => 'required|integer|min:1|max:10',
                'is_active' => 'nullable|in:on,1,true',
                'start_date' => 'nullable|date',
                'end_date' => 'nullable|date|after:start_date',
                'max_impressions' => 'nullable|integer|min:1',
            ]);

            $data = $request->except('media_file', '_token', '_method');
            $data['is_active'] = $request->has('is_active') ? 1 : 0;

            // Handle media upload
            if ($request->hasFile('media_file')) {
                // Delete old media file if exists
                $existingAd = \DB::table('ai_ads')->where('id', $id)->first();
                if ($existingAd && $existingAd->media_url) {
                    $oldFilename = basename($existingAd->media_url);
                    
                    // Try to delete from new assets location
                    $newPath = public_path('assets/images/ads/' . $oldFilename);
                    if (file_exists($newPath)) {
                        unlink($newPath);
                    }
                    
                    // Try to delete from old storage location (for migration compatibility)
                    $oldStoragePath = storage_path('app/public/ads/' . $oldFilename);
                    if (file_exists($oldStoragePath)) {
                        unlink($oldStoragePath);
                    }
                }
                
                $file = $request->file('media_file');
                $filename = time() . '_' . $file->getClientOriginalName();
                $file->move(public_path('assets/images/ads'), $filename);
                $data['media_url'] = asset('assets/images/ads/' . $filename);
            }

            // Handle targeting rules
            if ($request->has('targeting_rules')) {
                $data['targeting_rules'] = json_encode($request->targeting_rules);
            }

            // Add timestamp
            $data['updated_at'] = now();

            \DB::table('ai_ads')->where('id', $id)->update($data);
            \Log::info('Ad updated successfully', ['id' => $id]);

            return redirect()->route('admin.ads')->with('success', 'Ad updated successfully!');
        })->name('ads.update');
        Route::delete('/ads/{id}', [AdController::class, 'destroy'])->name('ads.delete');
        
        // Admin settings routes
        Route::get('/settings', [AdminController::class, 'settings'])->name('settings');
        Route::put('/settings', [AdminController::class, 'updateSettings'])->name('settings.update');
        Route::get('/branding', [AdminController::class, 'branding'])->name('branding');
        Route::put('/branding', [AdminController::class, 'updateBranding'])->name('branding.update');
        Route::post('/branding/upload', [AdminController::class, 'uploadBrandingImage'])->name('branding.upload');
        
        // Admin reports routes
        Route::get('/reports', [AdminController::class, 'reports'])->name('reports');
        
        // Admin notification routes
        Route::prefix('notifications')->name('notifications.')->group(function () {
            Route::get('/', [\App\Http\Controllers\Admin\NotificationController::class, 'index'])->name('index');
            Route::get('/create', [\App\Http\Controllers\Admin\NotificationController::class, 'create'])->name('create');
            Route::post('/', [\App\Http\Controllers\Admin\NotificationController::class, 'store'])->name('store');
            Route::get('/{notification}', [\App\Http\Controllers\Admin\NotificationController::class, 'show'])->name('show');
            Route::get('/{notification}/edit', [\App\Http\Controllers\Admin\NotificationController::class, 'edit'])->name('edit');
            Route::put('/{notification}', [\App\Http\Controllers\Admin\NotificationController::class, 'update'])->name('update');
            Route::post('/{notification}/send', [\App\Http\Controllers\Admin\NotificationController::class, 'send'])->name('send');
            Route::post('/{notification}/cancel', [\App\Http\Controllers\Admin\NotificationController::class, 'cancel'])->name('cancel');
            Route::delete('/{notification}', [\App\Http\Controllers\Admin\NotificationController::class, 'destroy'])->name('destroy');
            Route::get('/analytics/overview', [\App\Http\Controllers\Admin\NotificationController::class, 'analytics'])->name('analytics');
            Route::get('/subscriptions/manage', [\App\Http\Controllers\Admin\NotificationController::class, 'subscriptions'])->name('subscriptions');
            Route::post('/test/send', [\App\Http\Controllers\Admin\NotificationController::class, 'sendTest'])->name('test.send');
        });
        
        Route::post('/logout', [AdminController::class, 'logout'])->name('logout');
        
        
    });
});

// Test OneSignal page
Route::get('/test-onesignal', function () {
    return view('test-onesignal');
})->name('test.onesignal');

// Simple OneSignal test page
Route::get('/simple-onesignal-test', function () {
    return view('simple-onesignal-test');
})->name('simple.onesignal.test');

// Alternative OneSignal test page
Route::get('/alternative-onesignal-test', function () {
    return view('alternative-onesignal-test');
})->name('alternative.onesignal.test');

// API Routes for Ads (legacy - moved to api.php)
Route::prefix('api')->group(function () {
    Route::get('/ads', [AdController::class, 'getActiveAds'])->name('api.ads');
    Route::get('/ads/chatbot', [AdController::class, 'getActiveAds'])->name('api.ads.chatbot');
    Route::post('/ads/{id}/click', [AdController::class, 'trackClick'])->name('api.ads.click');
    
    // OneSignal API routes
    Route::prefix('onesignal')->group(function () {
        Route::post('/subscribe', [\App\Http\Controllers\Api\OneSignalController::class, 'subscribe'])->name('api.onesignal.subscribe');
        Route::post('/preferences', [\App\Http\Controllers\Api\OneSignalController::class, 'updatePreferences'])->name('api.onesignal.preferences');
        Route::post('/unsubscribe', [\App\Http\Controllers\Api\OneSignalController::class, 'unsubscribe'])->name('api.onesignal.unsubscribe');
        Route::get('/preferences', [\App\Http\Controllers\Api\OneSignalController::class, 'getPreferences'])->name('api.onesignal.get-preferences');
    });
    
    // Test notification endpoint
    Route::post('/test-notification', function (\Illuminate\Http\Request $request) {
        $request->validate([
            'player_id' => 'required|string',
            'title' => 'required|string',
            'message' => 'required|string',
        ]);
        
        try {
            $service = app(\App\Services\OneSignalService::class);
            
            $notificationData = [
                'title' => $request->title,
                'message' => $request->message,
                'data' => [
                    'type' => 'test_notification',
                    'timestamp' => now()->toISOString(),
                ],
            ];
            
            $result = $service->sendToPlayers([$request->player_id], $notificationData);
            
            if ($result && isset($result['id'])) {
                return response()->json([
                    'success' => true,
                    'message' => 'Test notification sent successfully!',
                    'onesignal_id' => $result['id'],
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to send test notification',
                ], 500);
            }
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage(),
            ], 500);
        }
    })->name('api.test-notification');
});







