{{-- FAB Chatbot - Persistent across all pages, reads page context --}}
<div id="chatFabContainer">
    <button id="chatFab" class="chat-fab-btn" title="AI Assistant - Ask me anything">
        <i class="fas fa-comments"></i>
    </button>
    <div id="chatFabPanel" class="chat-fab-panel">
        <div class="chat-fab-header">
            <div class="chat-fab-header-title">
                <i class="fas fa-robot"></i>
                <span>AI Assistant</span>
            </div>
            <div class="chat-fab-header-actions">
                @auth
                <button id="chatFabSave" class="chat-fab-icon-btn" title="Save this conversation">
                    <i class="fas fa-save"></i>
                </button>
                <button id="chatFabSessions" class="chat-fab-icon-btn" title="My saved conversations">
                    <i class="fas fa-history"></i>
                </button>
                @endauth
                <button id="chatFabClose" class="chat-fab-icon-btn" title="Close">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        </div>
        <div id="chatFabMessages" class="chat-fab-messages"></div>
        <div class="chat-fab-input-area">
            <input type="text" id="chatFabInput" placeholder="Ask about this page or anything..." autocomplete="off">
            <button id="chatFabSend" class="chat-fab-send-btn">
                <i class="fas fa-paper-plane"></i>
            </button>
        </div>
        <p class="chat-fab-hint">I can read this page and help explain, decipher, or find related content.</p>
    </div>
</div>

<style>
.chat-fab-btn {
    position: fixed;
    bottom: 24px;
    right: 24px;
    width: 56px;
    height: 56px;
    border-radius: 50%;
    background: linear-gradient(135deg, #0a57d4 0%, #2d72f0 100%);
    color: white;
    border: none;
    box-shadow: 0 4px 20px rgba(10, 87, 212, 0.4);
    cursor: pointer;
    z-index: 9998;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.25rem;
    transition: transform 0.2s, box-shadow 0.2s;
}
.chat-fab-btn:hover { transform: scale(1.05); box-shadow: 0 6px 24px rgba(10, 87, 212, 0.5); }
.chat-fab-panel {
    display: none;
    position: fixed;
    bottom: 90px;
    right: 24px;
    width: 380px;
    max-width: calc(100vw - 48px);
    height: 500px;
    max-height: calc(100vh - 140px);
    background: var(--bg-primary, #fff);
    border-radius: 16px;
    box-shadow: 0 8px 40px rgba(0,0,0,0.15);
    z-index: 9999;
    flex-direction: column;
    overflow: hidden;
    border: 1px solid var(--border-color, #dee2e6);
}
.chat-fab-panel.open { display: flex; }
.chat-fab-header {
    padding: 12px 16px;
    background: linear-gradient(135deg, #0a57d4 0%, #2d72f0 100%);
    color: white;
    display: flex;
    justify-content: space-between;
    align-items: center;
}
.chat-fab-header-title { display: flex; align-items: center; gap: 8px; font-weight: 600; }
.chat-fab-header-actions { display: flex; gap: 4px; }
.chat-fab-icon-btn {
    background: rgba(255,255,255,0.2);
    border: none;
    color: white;
    width: 32px;
    height: 32px;
    border-radius: 8px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
}
.chat-fab-icon-btn:hover { background: rgba(255,255,255,0.3); }
.chat-fab-messages {
    flex: 1;
    overflow-y: auto;
    padding: 16px;
    display: flex;
    flex-direction: column;
    gap: 12px;
}
.chat-fab-msg {
    max-width: 85%;
    padding: 10px 14px;
    border-radius: 12px;
    font-size: 0.9rem;
    line-height: 1.5;
}
.chat-fab-msg.user {
    align-self: flex-end;
    background: #0a57d4;
    color: white;
}
.chat-fab-msg.assistant {
    align-self: flex-start;
    background: var(--bg-secondary, #f8f9fa);
    color: var(--text-primary, #333);
    border: 1px solid var(--border-color, #dee2e6);
}
.chat-fab-msg.assistant a { color: #0a57d4; }
.chat-fab-input-area {
    padding: 12px 16px;
    display: flex;
    gap: 8px;
    border-top: 1px solid var(--border-color, #dee2e6);
}
.chat-fab-input-area input {
    flex: 1;
    padding: 10px 14px;
    border: 1px solid var(--border-color, #dee2e6);
    border-radius: 24px;
    font-size: 0.9rem;
    outline: none;
}
.chat-fab-input-area input:focus { border-color: #0a57d4; }
.chat-fab-send-btn {
    width: 44px;
    height: 44px;
    border-radius: 50%;
    background: #0a57d4;
    color: white;
    border: none;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
}
.chat-fab-send-btn:hover { background: #083a8f; }
.chat-fab-send-btn:disabled { opacity: 0.5; cursor: not-allowed; }
.chat-fab-hint {
    font-size: 0.75rem;
    color: var(--text-secondary, #666);
    padding: 0 16px 12px;
    margin: 0;
}
.chat-fab-loading { display: flex; gap: 4px; padding: 8px 0; }
.chat-fab-loading span { width: 6px; height: 6px; background: #0a57d4; border-radius: 50%; animation: chatFabDot 1s infinite; }
.chat-fab-loading span:nth-child(2) { animation-delay: 0.2s; }
.chat-fab-loading span:nth-child(3) { animation-delay: 0.4s; }
@keyframes chatFabDot {
    0%,100% { opacity: 0.4; transform: scale(1); }
    50% { opacity: 1; transform: scale(1.2); }
}
</style>

<script>
(function() {
    const STORAGE_KEY = 'mews_chat_messages';
    const SESSION_KEY = 'mews_chat_session_id';

    function getPageContext() {
        return {
            url: window.location.href,
            title: document.title,
            content: getPageContentSummary()
        };
    }

    function getPageContentSummary() {
        const main = document.querySelector('main') || document.querySelector('.app-content') || document.querySelector('.content') || document.body;
        if (!main) return '';
        const text = main.innerText || main.textContent || '';
        return text.replace(/\s+/g, ' ').trim().substring(0, 3000);
    }

    function getMessages() {
        try {
            const stored = sessionStorage.getItem(STORAGE_KEY);
            return stored ? JSON.parse(stored) : [];
        } catch (e) { return []; }
    }

    function saveMessages(messages) {
        sessionStorage.setItem(STORAGE_KEY, JSON.stringify(messages));
    }

    function getSessionId() {
        let id = sessionStorage.getItem(SESSION_KEY);
        if (!id) {
            id = 'fab_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
            sessionStorage.setItem(SESSION_KEY, id);
        }
        return id;
    }

    function renderMessages(messages) {
        const el = document.getElementById('chatFabMessages');
        el.innerHTML = messages.map(m => {
            const isUser = m.role === 'user';
            let content = (m.content || '').replace(/\n/g, '<br>');
            if (!isUser) {
                content = content.replace(/\[([^\]]+)\]\(([^)]+)\)/g, '<a href="$2" target="_blank">$1</a>');
                content = content.replace(/\*\*([^*]+)\*\*/g, '<strong>$1</strong>');
            }
            return `<div class="chat-fab-msg ${m.role}">${content}</div>`;
        }).join('');
        el.scrollTop = el.scrollHeight;
    }

    function addMessage(role, content) {
        const messages = getMessages();
        messages.push({ role, content });
        saveMessages(messages);
        renderMessages(messages);
        return messages;
    }

    function showLoading() {
        const el = document.getElementById('chatFabMessages');
        const div = document.createElement('div');
        div.id = 'chatFabLoading';
        div.className = 'chat-fab-msg assistant chat-fab-loading';
        div.innerHTML = '<span></span><span></span><span></span>';
        el.appendChild(div);
        el.scrollTop = el.scrollHeight;
    }

    function hideLoading() {
        document.getElementById('chatFabLoading')?.remove();
    }

    function sendMessage() {
        const input = document.getElementById('chatFabInput');
        const sendBtn = document.getElementById('chatFabSend');
        const query = input.value.trim();
        if (!query) return;

        input.value = '';
        sendBtn.disabled = true;
        addMessage('user', query);
        showLoading();

        const pageContext = getPageContext();
        const sessionId = getSessionId();
        const chatUrl = '{{ url("/api/chat") }}';

        fetch(chatUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: JSON.stringify({
                query,
                session_id: sessionId,
                page_context: pageContext
            })
        })
        .then(function(r) {
            if (!r.ok) {
                return r.text().then(function(text) {
                    try {
                        const data = JSON.parse(text);
                        throw { parsed: true, error: data.error || data.message || 'Request failed.' };
                    } catch (e) {
                        if (e.parsed) throw e;
                        throw { parsed: false, error: 'Server error. Please try again.' };
                    }
                });
            }
            return r.json();
        })
        .then(function(data) {
            hideLoading();
            sendBtn.disabled = false;
            if (data && data.success) {
                addMessage('assistant', data.response);
                if (data.news_data && data.news_data.length > 0) {
                    data.news_data.slice(0, 3).forEach(function(n) {
                        const title = n.post_title || n.title || 'Article';
                        const link = n.post_id ? '{{ url("/news/deep-dive") }}/' + n.post_id : '#';
                        addMessage('assistant', '📰 <a href="' + link + '">' + title + '</a>');
                    });
                }
            } else {
                addMessage('assistant', (data && data.error) ? data.error : 'Sorry, something went wrong.');
            }
        })
        .catch(function(err) {
            hideLoading();
            sendBtn.disabled = false;
            const msg = (err && err.error) ? err.error : 'Sorry, I couldn\'t connect. Please check your connection and try again.';
            addMessage('assistant', msg);
        });
    }

    function saveSession() {
        const messages = getMessages();
        if (messages.length === 0) {
            alert('No messages to save.');
            return;
        }
        const sessionsUrl = '{{ url("/api/chat/sessions") }}';
        const csrfToken = document.querySelector('meta[name="csrf-token"]') ? document.querySelector('meta[name="csrf-token"]').content : '';
        fetch(sessionsUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': csrfToken,
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: JSON.stringify({
                name: 'Chat ' + new Date().toLocaleString(),
                messages: messages,
                page_url: window.location.href,
                page_title: document.title
            })
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) alert('Conversation saved!');
            else alert(data.error || 'Failed to save.');
        })
        .catch(() => alert('Failed to save.'));
    }

    document.addEventListener('DOMContentLoaded', function() {
        const fab = document.getElementById('chatFab');
        const panel = document.getElementById('chatFabPanel');
        const closeBtn = document.getElementById('chatFabClose');
        const input = document.getElementById('chatFabInput');
        const sendBtn = document.getElementById('chatFabSend');
        const saveBtn = document.getElementById('chatFabSave');
        const sessionsBtn = document.getElementById('chatFabSessions');

        fab?.addEventListener('click', () => {
            panel.classList.toggle('open');
            if (panel.classList.contains('open')) {
                renderMessages(getMessages());
                input?.focus();
            }
        });
        closeBtn?.addEventListener('click', () => panel.classList.remove('open'));
        sendBtn?.addEventListener('click', sendMessage);
        input?.addEventListener('keypress', e => { if (e.key === 'Enter') sendMessage(); });
        saveBtn?.addEventListener('click', saveSession);

        if (sessionsBtn) {
            sessionsBtn.addEventListener('click', () => {
                window.location.href = '{{ route("user.dashboard") }}?tab=chats';
            });
        }

        renderMessages(getMessages());
    });
})();
</script>
