@extends('admin.layout')

@section('title', 'Edit Ad')

@section('content')
<div class="admin-header">
    <div class="d-flex justify-content-between align-items-center">
        <div>
            <h1 class="mb-0">
                <i class="fas fa-edit"></i> Edit Ad
            </h1>
            <p class="text-muted mb-0">Update your advertisement settings</p>
        </div>
        <a href="{{ route('admin.ads') }}" class="btn btn-outline-primary">
            <i class="fas fa-arrow-left"></i> Back to Ads
        </a>
    </div>
</div>

<div class="row">
    <div class="col-md-8">
        <div class="admin-card">
            <form action="{{ route('admin.ads.update', $ad->id) }}" method="POST" enctype="multipart/form-data">
                @csrf
                @method('PUT')
                
                <!-- Basic Information -->
                <h5 class="mb-3"><i class="fas fa-info-circle"></i> Basic Information</h5>
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="title" class="form-label">Ad Title *</label>
                            <input type="text" class="form-control @error('title') is-invalid @enderror" 
                                   id="title" name="title" value="{{ old('title', $ad->title) }}" required>
                            @error('title')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="label" class="form-label">Ad Label *</label>
                            <select class="form-select @error('label') is-invalid @enderror" id="label" name="label" required>
                                <option value="">Select Label</option>
                                <option value="Sponsored" {{ old('label', $ad->label) == 'Sponsored' ? 'selected' : '' }}>Sponsored</option>
                                <option value="Premium" {{ old('label', $ad->label) == 'Premium' ? 'selected' : '' }}>Premium</option>
                                <option value="Featured" {{ old('label', $ad->label) == 'Featured' ? 'selected' : '' }}>Featured</option>
                                <option value="Special Offer" {{ old('label', $ad->label) == 'Special Offer' ? 'selected' : '' }}>Special Offer</option>
                                <option value="Limited Time" {{ old('label', $ad->label) == 'Limited Time' ? 'selected' : '' }}>Limited Time</option>
                            </select>
                            @error('label')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <div class="mb-3">
                    <label for="content" class="form-label">Ad Content *</label>
                    <textarea class="form-control @error('content') is-invalid @enderror" 
                              id="content" name="content" rows="3" required>{{ old('content', $ad->content) }}</textarea>
                    <div class="form-text">Write engaging content that will appear in the ad</div>
                    @error('content')
                        <div class="invalid-feedback">{{ $message }}</div>
                    @enderror
                </div>

                <!-- Call to Action -->
                <h5 class="mb-3 mt-4"><i class="fas fa-bullseye"></i> Call to Action</h5>
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="cta_text" class="form-label">Button Text *</label>
                            <input type="text" class="form-control @error('cta_text') is-invalid @enderror" 
                                   id="cta_text" name="cta_text" value="{{ old('cta_text', $ad->cta_text) }}" required>
                            @error('cta_text')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="cta_url" class="form-label">Button URL *</label>
                            <input type="url" class="form-control @error('cta_url') is-invalid @enderror" 
                                   id="cta_url" name="cta_url" value="{{ old('cta_url', $ad->cta_url) }}" required>
                            @error('cta_url')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Media Upload -->
                <h5 class="mb-3 mt-4"><i class="fas fa-image"></i> Media (Optional)</h5>
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="media_type" class="form-label">Media Type</label>
                            <select class="form-select @error('media_type') is-invalid @enderror" 
                                    id="media_type" name="media_type" onchange="toggleMediaUpload()">
                                <option value="text" {{ old('media_type', $ad->media_type) == 'text' ? 'selected' : '' }}>Text Only</option>
                                <option value="image" {{ old('media_type', $ad->media_type) == 'image' ? 'selected' : '' }}>Image</option>
                                <option value="video" {{ old('media_type', $ad->media_type) == 'video' ? 'selected' : '' }}>Video</option>
                            </select>
                            @error('media_type')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="media_alt" class="form-label">Media Alt Text</label>
                            <input type="text" class="form-control @error('media_alt') is-invalid @enderror" 
                                   id="media_alt" name="media_alt" value="{{ old('media_alt', $ad->media_alt) }}">
                            <div class="form-text">Alternative text for accessibility</div>
                            @error('media_alt')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <div class="mb-3" id="media_upload_section" style="display: {{ in_array($ad->media_type, ['image', 'video']) ? 'block' : 'none' }};">
                    <label for="media_file" class="form-label">Upload New Media File</label>
                                                <input type="file" class="form-control @error('media_file') is-invalid @enderror" 
                                   id="media_file" name="media_file" 
                                   accept="{{ $ad->media_type == 'image' ? 'image/*' : 'video/*' }}">
                    <div class="form-text">
                        @if($ad->media_type == 'image')
                            Supported formats: JPG, PNG, GIF (max 10MB)
                        @elseif($ad->media_type == 'video')
                            Supported formats: MP4, AVI, MOV (max 10MB)
                        @endif
                    </div>
                    @error('media_file')
                        <div class="invalid-feedback">{{ $message }}</div>
                    @enderror
                    
                    @if($ad->media_url)
                        <div class="mt-3">
                            <label class="form-label">Current Media:</label>
                            @if($ad->media_type == 'image')
                                <img src="{{ $ad->media_url }}" alt="{{ $ad->media_alt }}" 
                                     class="img-thumbnail d-block" style="max-width: 200px;">
                            @elseif($ad->media_type == 'video')
                                <video controls class="img-thumbnail d-block" style="max-width: 200px;">
                                    <source src="{{ $ad->media_url }}" type="video/mp4">
                                    Your browser does not support the video tag.
                                </video>
                            @endif
                            <small class="text-muted">Upload a new file to replace this media</small>
                        </div>
                    @endif
                </div>

                <!-- Display Settings -->
                <h5 class="mb-3 mt-4"><i class="fas fa-sliders-h"></i> Display Settings</h5>
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="display_duration" class="form-label">Display Duration (ms) *</label>
                            <input type="number" class="form-control @error('display_duration') is-invalid @enderror" 
                                   id="display_duration" name="display_duration" 
                                   value="{{ old('display_duration', $ad->display_duration) }}" 
                                   min="0" max="10000" required>
                            <div class="form-text">
                                <strong>0 = Stay until AI responds</strong> (recommended)<br>
                                500-10000ms = Auto-remove after specified time
                            </div>
                            @error('display_duration')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="rotation_weight" class="form-label">Rotation Weight *</label>
                            <select class="form-select @error('rotation_weight') is-invalid @enderror" 
                                    id="rotation_weight" name="rotation_weight" required>
                                @for($i = 1; $i <= 10; $i++)
                                    <option value="{{ $i }}" {{ old('rotation_weight', $ad->rotation_weight) == $i ? 'selected' : '' }}>
                                        {{ $i }} {{ $i == 1 ? 'time' : 'times' }}
                                    </option>
                                @endfor
                            </select>
                            <div class="form-text">Higher weight = more frequent display</div>
                            @error('rotation_weight')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Scheduling -->
                <h5 class="mb-3 mt-4"><i class="fas fa-calendar"></i> Scheduling (Optional)</h5>
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="start_date" class="form-label">Start Date</label>
                            <input type="datetime-local" class="form-control @error('start_date') is-invalid @enderror" 
                                   id="start_date" name="start_date" 
                                   value="{{ old('start_date', $ad->start_date ? \Carbon\Carbon::parse($ad->start_date)->format('Y-m-d\TH:i') : '') }}">
                            <div class="form-text">When to start showing this ad</div>
                            @error('start_date')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="end_date" class="form-label">End Date</label>
                            <input type="datetime-local" class="form-control @error('end_date') is-invalid @enderror" 
                                   id="end_date" name="end_date" 
                                   value="{{ old('end_date', $ad->end_date ? \Carbon\Carbon::parse($ad->end_date)->format('Y-m-d\TH:i') : '') }}">
                            <div class="form-text">When to stop showing this ad</div>
                            @error('end_date')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Targeting -->
                <h5 class="mb-3 mt-4"><i class="fas fa-crosshairs"></i> Targeting (Optional)</h5>
                <div class="mb-3">
                    <label for="max_impressions" class="form-label">Maximum Impressions</label>
                                                <input type="number" class="form-control @error('max_impressions') is-invalid @enderror" 
                                   id="max_impressions" name="max_impressions" 
                                   value="{{ old('max_impressions', $ad->max_impressions) }}" min="1">
                    <div class="form-text">Stop showing after this many impressions (leave empty for unlimited)</div>
                    @error('max_impressions')
                        <div class="invalid-feedback">{{ $message }}</div>
                    @enderror
                </div>

                <!-- Status -->
                <div class="mb-3">
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" id="is_active" name="is_active" 
                               {{ old('is_active', $ad->is_active) ? 'checked' : '' }}>
                        <label class="form-check-label" for="is_active">
                            Active - Show this ad to users
                        </label>
                    </div>
                </div>

                <!-- Submit Buttons -->
                <div class="d-flex gap-2 mt-4">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> Update Ad
                    </button>
                    <a href="{{ route('admin.ads') }}" class="btn btn-outline-secondary">
                        <i class="fas fa-times"></i> Cancel
                    </a>
                </div>
            </form>
        </div>
    </div>

    <div class="col-md-4">
        <!-- Ad Preview -->
        <div class="admin-card">
            <h5><i class="fas fa-eye"></i> Ad Preview</h5>
            <div id="adPreview" class="border rounded p-3 bg-light">
                <div class="text-center">
                    <h6 id="previewTitle">{{ $ad->title }}</h6>
                    <p id="previewContent" class="text-muted">{{ $ad->content }}</p>
                    <button class="btn btn-sm btn-primary" id="previewCta">{{ $ad->cta_text }}</button>
                </div>
            </div>
        </div>

        <!-- Performance Stats -->
        <div class="admin-card">
            <h5><i class="fas fa-chart-line"></i> Performance</h5>
            <div class="row text-center">
                <div class="col-6">
                    <h4 class="text-primary">{{ number_format($ad->current_impressions) }}</h4>
                    <small class="text-muted">Impressions</small>
                </div>
                <div class="col-6">
                    <h4 class="text-success">{{ number_format($ad->clicks) }}</h4>
                    <small class="text-muted">Clicks</small>
                </div>
            </div>
            @if($ad->current_impressions > 0)
                <div class="mt-3">
                    <strong>CTR:</strong> {{ number_format(($ad->clicks / $ad->current_impressions) * 100, 2) }}%
                </div>
            @endif
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
function toggleMediaUpload() {
    const mediaType = document.getElementById('media_type').value;
    const mediaSection = document.getElementById('media_upload_section');
    
    if (mediaType === 'text') {
        mediaSection.style.display = 'none';
    } else {
        mediaSection.style.display = 'block';
    }
}

// Live preview updates
document.getElementById('title').addEventListener('input', function() {
    document.getElementById('previewTitle').textContent = this.value;
});

document.getElementById('content').addEventListener('input', function() {
    document.getElementById('previewContent').textContent = this.value;
});

document.getElementById('cta_text').addEventListener('input', function() {
    document.getElementById('previewCta').textContent = this.value;
});

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    toggleMediaUpload();
});
</script>
@endsection
