@extends('admin.layout')

@section('title', 'Create New Ad')

@section('content')
<div class="admin-header">
    <div class="d-flex justify-content-between align-items-center">
        <div>
            <h1 class="mb-0">
                <i class="fas fa-plus-circle"></i> Create New Ad
            </h1>
            <p class="text-muted mb-0">Design engaging advertisements for your chatbot</p>
        </div>
        <a href="{{ route('admin.ads') }}" class="btn btn-outline-primary">
            <i class="fas fa-arrow-left"></i> Back to Ads
        </a>
    </div>
</div>

<div class="row">
    <div class="col-md-8">
        <div class="admin-card">
            <form action="{{ route('admin.ads.store') }}" method="POST" enctype="multipart/form-data">
                @csrf
                
                <!-- Basic Information -->
                <h5 class="mb-3"><i class="fas fa-info-circle"></i> Basic Information</h5>
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="title" class="form-label">Ad Title *</label>
                            <input type="text" class="form-control @error('title') is-invalid @enderror" 
                                   id="title" name="title" value="{{ old('title') }}" required>
                            @error('title')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="label" class="form-label">Ad Label *</label>
                            <select class="form-select @error('label') is-invalid @enderror" id="label" name="label" required>
                                <option value="">Select Label</option>
                                <option value="Sponsored" {{ old('label') == 'Sponsored' ? 'selected' : '' }}>Sponsored</option>
                                <option value="Premium" {{ old('label') == 'Premium' ? 'selected' : '' }}>Premium</option>
                                <option value="Featured" {{ old('label') == 'Featured' ? 'selected' : '' }}>Featured</option>
                                <option value="Special Offer" {{ old('label') == 'Special Offer' ? 'selected' : '' }}>Special Offer</option>
                                <option value="Limited Time" {{ old('label') == 'Limited Time' ? 'selected' : '' }}>Limited Time</option>
                            </select>
                            @error('label')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <div class="mb-3">
                    <label for="content" class="form-label">Ad Content *</label>
                    <textarea class="form-control @error('content') is-invalid @enderror" 
                              id="content" name="content" rows="3" required>{{ old('content') }}</textarea>
                    <div class="form-text">Write engaging content that will appear in the ad</div>
                    @error('content')
                        <div class="invalid-feedback">{{ $message }}</div>
                    @enderror
                </div>

                <!-- Call to Action -->
                <h5 class="mb-3 mt-4"><i class="fas fa-bullseye"></i> Call to Action</h5>
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="cta_text" class="form-label">Button Text *</label>
                            <input type="text" class="form-control @error('cta_text') is-invalid @enderror" 
                                   id="cta_text" name="cta_text" value="{{ old('cta_text') }}" required>
                            @error('cta_text')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="cta_url" class="form-label">Button URL *</label>
                            <input type="url" class="form-control @error('cta_url') is-invalid @enderror" 
                                   id="cta_url" name="cta_url" value="{{ old('cta_url') }}" required>
                            @error('cta_url')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Media Upload -->
                <h5 class="mb-3 mt-4"><i class="fas fa-image"></i> Media (Optional)</h5>
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="media_type" class="form-label">Media Type</label>
                            <select class="form-select @error('media_type') is-invalid @enderror" 
                                    id="media_type" name="media_type" onchange="toggleMediaUpload()">
                                <option value="text" {{ old('media_type') == 'text' ? 'selected' : '' }}>Text Only</option>
                                <option value="image" {{ old('media_type') == 'image' ? 'selected' : '' }}>Image</option>
                                <option value="video" {{ old('media_type') == 'video' ? 'selected' : '' }}>Video</option>
                            </select>
                            @error('media_type')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3" id="media_upload_section" style="display: none;">
                            <label for="media_file" class="form-label">Upload Media</label>
                            <input type="file" class="form-control @error('media_file') is-invalid @enderror" 
                                   id="media_file" name="media_file" accept="image/*,video/*">
                            <div class="form-text" id="media_help"></div>
                            @error('media_file')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <div class="mb-3" id="media_alt_section" style="display: none;">
                    <label for="media_alt" class="form-label">Media Alt Text</label>
                    <input type="text" class="form-control @error('media_alt') is-invalid @enderror" 
                           id="media_alt" name="media_alt" value="{{ old('media_alt') }}">
                    <div class="form-text">Description for accessibility and SEO</div>
                    @error('media_alt')
                        <div class="invalid-feedback">{{ $message }}</div>
                    @enderror
                </div>

                <!-- Display Settings -->
                <h5 class="mb-3 mt-4"><i class="fas fa-sliders-h"></i> Display Settings</h5>
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="display_duration" class="form-label">Display Duration (ms) *</label>
                            <input type="number" class="form-control @error('display_duration') is-invalid @enderror" 
                                   id="display_duration" name="display_duration" 
                                   value="{{ old('display_duration', 0) }}" min="0" max="10000" required>
                            <div class="form-text">
                                <strong>0 = Stay until AI responds</strong> (recommended)<br>
                                500-10000ms = Auto-remove after specified time
                            </div>
                            @error('display_duration')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="rotation_weight" class="form-label">Rotation Weight *</label>
                            <select class="form-select @error('rotation_weight') is-invalid @enderror" 
                                    id="rotation_weight" name="rotation_weight" required>
                                <option value="1" {{ old('rotation_weight') == 1 ? 'selected' : '' }}>1 - Low Priority</option>
                                <option value="2" {{ old('rotation_weight') == 2 ? 'selected' : '' }}>2</option>
                                <option value="3" {{ old('rotation_weight') == 3 ? 'selected' : '' }}>3</option>
                                <option value="5" {{ old('rotation_weight') == 5 ? 'selected' : '' }}>5 - Medium Priority</option>
                                <option value="7" {{ old('rotation_weight') == 7 ? 'selected' : '' }}>7</option>
                                <option value="10" {{ old('rotation_weight') == 10 ? 'selected' : '' }}>10 - High Priority</option>
                            </select>
                            <div class="form-text">Higher weight = more frequent display</div>
                            @error('rotation_weight')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Scheduling -->
                <h5 class="mb-3 mt-4"><i class="fas fa-calendar"></i> Scheduling (Optional)</h5>
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="start_date" class="form-label">Start Date</label>
                            <input type="datetime-local" class="form-control @error('start_date') is-invalid @enderror" 
                                   id="start_date" name="start_date" value="{{ old('start_date') }}">
                            @error('start_date')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="end_date" class="form-label">End Date</label>
                            <input type="datetime-local" class="form-control @error('end_date') is-invalid @enderror" 
                                   id="end_date" name="end_date" value="{{ old('end_date') }}">
                            @error('end_date')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Limits -->
                <h5 class="mb-3 mt-4"><i class="fas fa-chart-line"></i> Performance Limits (Optional)</h5>
                <div class="mb-3">
                    <label for="max_impressions" class="form-label">Maximum Impressions</label>
                    <input type="number" class="form-control @error('max_impressions') is-invalid @enderror" 
                           id="max_impressions" name="max_impressions" 
                           value="{{ old('max_impressions') }}" min="1">
                    <div class="form-text">Leave empty for unlimited impressions</div>
                    @error('max_impressions')
                        <div class="invalid-feedback">{{ $message }}</div>
                    @enderror
                </div>

                <!-- Status -->
                <div class="mb-3">
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" id="is_active" name="is_active" 
                               {{ old('is_active', true) ? 'checked' : '' }}>
                        <label class="form-check-label" for="is_active">
                            Activate this ad immediately
                        </label>
                    </div>
                </div>

                <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                    <a href="{{ route('admin.ads') }}" class="btn btn-outline-secondary me-md-2">Cancel</a>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> Create Ad
                    </button>
                </div>
            </form>
        </div>
    </div>

    <div class="col-md-4">
        <!-- Preview Card -->
        <div class="admin-card">
            <h5><i class="fas fa-eye"></i> Ad Preview</h5>
            <div id="ad_preview" class="border rounded p-3 bg-light">
                <div class="text-center">
                    <i class="fas fa-ad fa-2x text-muted mb-2"></i>
                    <p class="text-muted">Fill out the form to see a preview</p>
                </div>
            </div>
        </div>

        <!-- Tips -->
        <div class="admin-card">
            <h5><i class="fas fa-lightbulb"></i> Tips for Better Ads</h5>
            <ul class="list-unstyled">
                <li class="mb-2">
                    <i class="fas fa-check text-success me-2"></i>
                    Keep content concise and engaging
                </li>
                <li class="mb-2">
                    <i class="fas fa-check text-success me-2"></i>
                    Use clear call-to-action buttons
                </li>
                <li class="mb-2">
                    <i class="fas fa-check text-success me-2"></i>
                    Test different rotation weights
                </li>
                <li class="mb-2">
                    <i class="fas fa-check text-success me-2"></i>
                    Monitor performance metrics
                </li>
                <li class="mb-2">
                    <i class="fas fa-check text-success me-2"></i>
                    Use high-quality media files
                </li>
            </ul>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
function toggleMediaUpload() {
    const mediaType = document.getElementById('media_type').value;
    const mediaUploadSection = document.getElementById('media_upload_section');
    const mediaAltSection = document.getElementById('media_alt_section');
    const mediaFile = document.getElementById('media_file');
    const mediaHelp = document.getElementById('media_help');

    if (mediaType === 'text') {
        mediaUploadSection.style.display = 'none';
        mediaAltSection.style.display = 'none';
        mediaFile.required = false;
    } else {
        mediaUploadSection.style.display = 'block';
        mediaAltSection.style.display = 'block';
        mediaFile.required = true;
        
        if (mediaType === 'image') {
            mediaFile.accept = 'image/*';
            mediaHelp.textContent = 'Supported: JPG, PNG, GIF (max 2MB)';
        } else if (mediaType === 'video') {
            mediaFile.accept = 'video/*';
            mediaHelp.textContent = 'Supported: MP4, AVI, MOV (max 10MB)';
        }
    }
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    toggleMediaUpload();
    updatePreview(); // Initialize preview
    
    // Debug form submission
    const form = document.querySelector('form');
    if (form) {
        form.addEventListener('submit', function(e) {
            console.log('Form submitting to:', form.action);
            console.log('Form method:', form.method);
            console.log('Form data:', new FormData(form));
            console.log('Form validation passed, submitting...');
        });
    }
});

// Live preview updates
document.getElementById('title').addEventListener('input', updatePreview);
document.getElementById('content').addEventListener('input', updatePreview);
document.getElementById('cta_text').addEventListener('input', updatePreview);
document.getElementById('label').addEventListener('change', updatePreview);
document.getElementById('media_type').addEventListener('change', updatePreview);
document.getElementById('media_file').addEventListener('change', handleMediaPreview);

function handleMediaPreview(event) {
    const file = event.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            window.mediaPreviewUrl = e.target.result;
            updatePreview();
        };
        reader.readAsDataURL(file);
    } else {
        window.mediaPreviewUrl = null;
        updatePreview();
    }
}

function updatePreview() {
    const title = document.getElementById('title').value || 'Ad Title';
    const content = document.getElementById('content').value || 'Ad content will appear here...';
    const ctaText = document.getElementById('cta_text').value || 'Learn More';
    const label = document.getElementById('label').value || 'Sponsored';
    const mediaType = document.getElementById('media_type').value || 'text';
    
    let mediaHtml = '';
    if (mediaType === 'image' && window.mediaPreviewUrl) {
        mediaHtml = `<img src="${window.mediaPreviewUrl}" alt="Preview" style="width: 100%; max-height: 200px; object-fit: cover; border-radius: 8px; margin-bottom: 10px;">`;
    } else if (mediaType === 'video' && window.mediaPreviewUrl) {
        mediaHtml = `<video controls style="width: 100%; max-height: 200px; border-radius: 8px; margin-bottom: 10px;"><source src="${window.mediaPreviewUrl}" type="video/mp4"></video>`;
    }
    
    const preview = document.getElementById('ad_preview');
    preview.innerHTML = `
        <div class="ad-preview">
            <h6 class="text-muted mb-2">${label}</h6>
            ${mediaHtml}
            <h6 class="fw-bold mb-2">${title}</h6>
            <p class="mb-3">${content}</p>
            <button class="btn btn-primary btn-sm">${ctaText}</button>
        </div>
    `;
}
</script>
@endsection
