@extends('admin.layout')

@section('title', 'Ads Management')

@section('content')
<div class="admin-header">
    <div class="d-flex justify-content-between align-items-center">
        <div>
            <h1 class="mb-0">
                <i class="fas fa-ad"></i> Ads Management
            </h1>
            <p class="text-muted mb-0">Create, edit, and manage your chatbot advertisements</p>
        </div>
        <a href="{{ route('admin.ads.create') }}" class="btn btn-primary">
            <i class="fas fa-plus"></i> Create New Ad
        </a>
    </div>
</div>

<!-- Ads Table -->
<div class="admin-card">
    @if($ads->count() > 0)
        <div class="table-responsive">
            <table class="table table-hover">
                <thead>
                    <tr>
                        <th>Ad Details</th>
                        <th>Media</th>
                        <th>Targeting</th>
                        <th>Performance</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($ads as $ad)
                    <tr>
                        <td>
                            <div>
                                <strong>{{ $ad->title }}</strong>
                                <br>
                                <small class="text-muted">{{ Str::limit($ad->content, 60) }}</small>
                                <br>
                                <span class="badge bg-secondary">{{ $ad->label }}</span>
                                <span class="badge bg-info">{{ ucfirst($ad->media_type) }}</span>
                            </div>
                        </td>
                        <td>
                            @if($ad->media_type === 'image' && $ad->media_url)
                                <img src="{{ $ad->media_url }}" alt="{{ $ad->media_alt }}" 
                                     style="width: 60px; height: 40px; object-fit: cover; border-radius: 5px;">
                            @elseif($ad->media_type === 'video' && $ad->media_url)
                                <i class="fas fa-video fa-2x text-primary"></i>
                            @else
                                <i class="fas fa-text-height fa-2x text-muted"></i>
                            @endif
                        </td>
                        <td>
                            <div>
                                <small class="text-muted">Weight: {{ $ad->rotation_weight }}</small><br>
                                <small class="text-muted">Duration: {{ $ad->display_duration }}ms</small>
                                @if($ad->start_date || $ad->end_date)
                                    <br>
                                    <small class="text-muted">
                                        @if($ad->start_date) From: {{ \Carbon\Carbon::parse($ad->start_date)->format('M d') }} @endif
                                        @if($ad->end_date) To: {{ \Carbon\Carbon::parse($ad->end_date)->format('M d') }} @endif
                                    </small>
                                @endif
                            </div>
                        </td>
                        <td>
                            <div>
                                <strong>{{ number_format($ad->current_impressions) }}</strong> impressions<br>
                                <strong>{{ number_format($ad->clicks) }}</strong> clicks
                                @if($ad->current_impressions > 0)
                                    <br>
                                    <small class="text-muted">
                                        CTR: {{ number_format(($ad->clicks / $ad->current_impressions) * 100, 2) }}%
                                    </small>
                                @endif
                            </div>
                        </td>
                        <td>
                            @if($ad->is_active)
                                <span class="badge bg-success">Active</span>
                            @else
                                <span class="badge bg-secondary">Inactive</span>
                            @endif
                            @if($ad->max_impressions && $ad->current_impressions >= $ad->max_impressions)
                                <br><span class="badge bg-warning">Limit Reached</span>
                            @endif
                        </td>
                        <td>
                            <div class="btn-group" role="group">
                                <a href="{{ route('admin.ads.edit', $ad->id) }}" 
                                   class="btn btn-sm btn-outline-primary" title="Edit">
                                    <i class="fas fa-edit"></i>
                                </a>
                                <button type="button" class="btn btn-sm btn-outline-danger" 
                                        onclick="deleteAd({{ $ad->id }})" title="Delete">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    @else
        <div class="text-center py-5">
            <i class="fas fa-ad fa-4x text-muted mb-4"></i>
            <h4 class="text-muted">No ads created yet</h4>
            <p class="text-muted">Start monetizing your chatbot by creating engaging advertisements</p>
            <a href="{{ route('admin.ads.create') }}" class="btn btn-primary btn-lg">
                <i class="fas fa-plus"></i> Create Your First Ad
            </a>
        </div>
    @endif
</div>

<!-- Performance Summary -->
@if($ads->count() > 0)
<div class="admin-card">
    <h5><i class="fas fa-chart-bar"></i> Performance Summary</h5>
    <div class="row">
        <div class="col-md-3">
            <div class="text-center">
                <h4 class="text-primary">{{ $ads->where('is_active', true)->count() }}</h4>
                <small class="text-muted">Active Campaigns</small>
            </div>
        </div>
        <div class="col-md-3">
            <div class="text-center">
                <h4 class="text-success">{{ number_format($ads->sum('current_impressions')) }}</h4>
                <small class="text-muted">Total Impressions</small>
            </div>
        </div>
        <div class="col-md-3">
            <div class="text-center">
                <h4 class="text-info">{{ number_format($ads->sum('clicks')) }}</h4>
                <small class="text-muted">Total Clicks</small>
            </div>
        </div>
        <div class="col-md-3">
            <div class="text-center">
                @php
                    $totalImpressions = $ads->sum('current_impressions');
                    $totalClicks = $ads->sum('clicks');
                    $ctr = $totalImpressions > 0 ? ($totalClicks / $totalImpressions) * 100 : 0;
                @endphp
                <h4 class="text-warning">{{ number_format($ctr, 2) }}%</h4>
                <small class="text-muted">Overall CTR</small>
            </div>
        </div>
    </div>
</div>
@endif
@endsection

@section('scripts')
<script>
function deleteAd(adId) {
    if (confirm('Are you sure you want to delete this ad? This action cannot be undone.')) {
        // Create a form to submit the delete request
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = '{{ route("admin.ads.delete", ":id") }}'.replace(':id', adId);
        
        // Add CSRF token
        const csrfToken = document.createElement('input');
        csrfToken.type = 'hidden';
        csrfToken.name = '_token';
        csrfToken.value = '{{ csrf_token() }}';
        form.appendChild(csrfToken);
        
        // Add method override
        const methodField = document.createElement('input');
        methodField.type = 'hidden';
        methodField.name = '_method';
        methodField.value = 'DELETE';
        form.appendChild(methodField);
        
        document.body.appendChild(form);
        form.submit();
    }
}
</script>
@endsection


