<?php

namespace App\Services;

use App\Models\NotificationCampaign;
use App\Models\UserNotificationSubscription;
use App\Models\NotificationLog;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class OneSignalService
{
    protected $appId;
    protected $restApiKey;
    protected $baseUrl = 'https://onesignal.com/api/v1';

    public function __construct()
    {
        $this->appId = config('services.onesignal.app_id');
        $this->restApiKey = config('services.onesignal.rest_api_key');
    }

    /**
     * Send notification to specific players.
     */
    public function sendToPlayers(array $playerIds, array $notificationData)
    {
        $payload = [
            'app_id' => $this->appId,
            'include_player_ids' => $playerIds,
            'headings' => ['en' => $notificationData['title']],
            'contents' => ['en' => $notificationData['message']],
            'data' => $notificationData['data'] ?? [],
            'url' => $notificationData['action_url'] ?? null,
            'web_buttons' => $notificationData['web_buttons'] ?? [],
            'chrome_web_icon' => $notificationData['icon'] ?? null,
            'big_picture' => $notificationData['image_url'] ?? null,
            'send_after' => $notificationData['send_after'] ?? null,
        ];

        return $this->makeRequest('notifications', $payload);
    }

    /**
     * Send notification to segments.
     */
    public function sendToSegments(array $segments, array $notificationData)
    {
        $payload = [
            'app_id' => $this->appId,
            'included_segments' => $segments,
            'headings' => ['en' => $notificationData['title']],
            'contents' => ['en' => $notificationData['message']],
            'data' => $notificationData['data'] ?? [],
            'url' => $notificationData['action_url'] ?? null,
            'web_buttons' => $notificationData['web_buttons'] ?? [],
            'chrome_web_icon' => $notificationData['icon'] ?? null,
            'big_picture' => $notificationData['image_url'] ?? null,
            'send_after' => $notificationData['send_after'] ?? null,
        ];

        return $this->makeRequest('notifications', $payload);
    }

    /**
     * Send notification to all users.
     */
    public function sendToAll(array $notificationData)
    {
        return $this->sendToSegments(['All'], $notificationData);
    }

    /**
     * Send campaign notification.
     */
    public function sendCampaign(NotificationCampaign $campaign)
    {
        try {
            // Get target subscriptions
            $subscriptions = $this->getTargetSubscriptions($campaign);
            
            if ($subscriptions->isEmpty()) {
                Log::warning('No target subscriptions found for campaign', ['campaign_id' => $campaign->id]);
                return false;
            }

            $playerIds = $subscriptions->pluck('onesignal_player_id')->toArray();
            
            // Prepare notification data
            $notificationData = [
                'title' => $campaign->title,
                'message' => $campaign->message,
                'icon' => $campaign->icon,
                'image_url' => $campaign->image_url,
                'action_url' => $campaign->action_url,
                'data' => [
                    'campaign_id' => $campaign->id,
                    'type' => 'news_notification',
                    'utm_source' => 'onesignal',
                    'utm_medium' => 'push',
                    'utm_campaign' => 'news_' . $campaign->id,
                ],
                'web_buttons' => $campaign->action_text ? [
                    [
                        'id' => 'read_more',
                        'text' => $campaign->action_text,
                        'icon' => 'https://via.placeholder.com/24x24/ff6b35/ffffff?text=→',
                        'url' => $campaign->action_url,
                    ]
                ] : [],
            ];

            // Send notification
            $response = $this->sendToPlayers($playerIds, $notificationData);
            
            if ($response && isset($response['id'])) {
                // Update campaign
                $campaign->update([
                    'status' => 'sent',
                    'onesignal_campaign_id' => $response['id'],
                    'sent_count' => $response['recipients'] ?? count($playerIds),
                ]);

                // Create notification logs
                $this->createNotificationLogs($campaign, $subscriptions, $response['id']);
                
                Log::info('Campaign sent successfully', [
                    'campaign_id' => $campaign->id,
                    'onesignal_id' => $response['id'],
                    'recipients' => $response['recipients'] ?? count($playerIds),
                ]);

                return true;
            }

            return false;

        } catch (\Exception $e) {
            Log::error('Failed to send campaign', [
                'campaign_id' => $campaign->id,
                'error' => $e->getMessage(),
            ]);
            
            return false;
        }
    }

    /**
     * Get target subscriptions for a campaign.
     */
    protected function getTargetSubscriptions(NotificationCampaign $campaign)
    {
        $query = UserNotificationSubscription::active();

        // Filter by target audience
        if ($campaign->target_audience === 'registered') {
            $query->whereNotNull('user_id');
        } elseif ($campaign->target_audience === 'premium') {
            $query->whereHas('user', function ($q) {
                $q->where('is_premium', true);
            });
        }

        // Filter by categories
        if (!empty($campaign->target_categories)) {
            $query->where(function ($q) use ($campaign) {
                foreach ($campaign->target_categories as $category) {
                    $q->orWhereJsonContains('subscribed_categories', $category);
                }
            });
        }

        // Filter by locations
        if (!empty($campaign->target_locations)) {
            $query->where(function ($q) use ($campaign) {
                foreach ($campaign->target_locations as $location) {
                    $q->orWhereJsonContains('subscribed_locations', $location);
                }
            });
        }

        // Filter by interests
        if (!empty($campaign->target_interests)) {
            $query->where(function ($q) use ($campaign) {
                foreach ($campaign->target_interests as $interest) {
                    $q->orWhereJsonContains('subscribed_interests', $interest);
                }
            });
        }

        return $query->get();
    }

    /**
     * Create notification logs for sent notifications.
     */
    protected function createNotificationLogs(NotificationCampaign $campaign, $subscriptions, string $onesignalId)
    {
        $logs = [];
        
        foreach ($subscriptions as $subscription) {
            $logs[] = [
                'campaign_id' => $campaign->id,
                'subscription_id' => $subscription->id,
                'user_id' => $subscription->user_id,
                'onesignal_player_id' => $subscription->onesignal_player_id,
                'onesignal_notification_id' => $onesignalId,
                'status' => 'sent',
                'sent_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        NotificationLog::insert($logs);
    }

    /**
     * Get notification delivery status.
     */
    public function getNotificationStatus(string $notificationId)
    {
        return $this->makeRequest("notifications/{$notificationId}");
    }

    /**
     * Create or update player.
     */
    public function createPlayer(array $playerData)
    {
        $payload = array_merge([
            'app_id' => $this->appId,
        ], $playerData);

        return $this->makeRequest('players', $payload, 'POST');
    }

    /**
     * Update player.
     */
    public function updatePlayer(string $playerId, array $playerData)
    {
        return $this->makeRequest("players/{$playerId}", $playerData, 'PUT');
    }

    /**
     * Get player information.
     */
    public function getPlayer(string $playerId)
    {
        return $this->makeRequest("players/{$playerId}");
    }

    /**
     * Make HTTP request to OneSignal API.
     */
    public function makeRequest(string $endpoint, array $data = [], string $method = 'POST')
    {
        $url = "{$this->baseUrl}/{$endpoint}";
        
        // Always include app_id in the request body for OneSignal API
        $data['app_id'] = $this->appId;
        
        $response = Http::withHeaders([
            'Authorization' => "Basic {$this->restApiKey}",
            'Content-Type' => 'application/json',
        ])->$method($url, $data);

        if ($response->successful()) {
            return $response->json();
        }

        Log::error('OneSignal API request failed', [
            'endpoint' => $endpoint,
            'method' => $method,
            'status' => $response->status(),
            'response' => $response->body(),
            'request_data' => $data,
        ]);

        return null;
    }

    /**
     * Send breaking news notification.
     */
    public function sendBreakingNews(string $title, string $message, string $url = null)
    {
        $subscriptions = UserNotificationSubscription::active()
            ->where('breaking_news_enabled', true)
            ->get();

        if ($subscriptions->isEmpty()) {
            return false;
        }

        $playerIds = $subscriptions->pluck('onesignal_player_id')->toArray();

        $notificationData = [
            'title' => "🚨 BREAKING: {$title}",
            'message' => $message,
            'action_url' => $url,
            'data' => [
                'type' => 'breaking_news',
                'priority' => 'high',
            ],
            'web_buttons' => $url ? [
                [
                    'id' => 'read_now',
                    'text' => 'Read Now',
                    'icon' => 'https://via.placeholder.com/24x24/ff6b35/ffffff?text=📰',
                    'url' => $url,
                ]
            ] : [],
        ];

        return $this->sendToPlayers($playerIds, $notificationData);
    }

    /**
     * Send daily digest notification.
     */
    public function sendDailyDigest()
    {
        $subscriptions = UserNotificationSubscription::active()
            ->where('daily_digest_enabled', true)
            ->get();

        if ($subscriptions->isEmpty()) {
            return false;
        }

        $playerIds = $subscriptions->pluck('onesignal_player_id')->toArray();

        $notificationData = [
            'title' => '📰 Your Daily News Digest',
            'message' => 'Stay updated with today\'s top stories and breaking news.',
            'action_url' => route('news.chatbot'),
            'data' => [
                'type' => 'daily_digest',
                'priority' => 'normal',
            ],
            'web_buttons' => [
                [
                    'id' => 'read_digest',
                    'text' => 'Read Digest',
                    'icon' => 'https://via.placeholder.com/24x24/ff6b35/ffffff?text=📰',
                    'url' => route('news.chatbot'),
                ]
            ],
        ];

        return $this->sendToPlayers($playerIds, $notificationData);
    }

    /**
     * Get app information from OneSignal.
     */
    public function getAppInfo()
    {
        return $this->makeRequest("apps/{$this->appId}", [], 'GET');
    }

    /**
     * Get player count from OneSignal.
     */
    public function getPlayerCount()
    {
        $result = $this->makeRequest("players?app_id={$this->appId}&limit=1", [], 'GET');
        return $result['total_count'] ?? 0;
    }

    /**
     * Send breaking news notification to all users.
     */
    public function sendBreakingNewsToAll(string $title, string $message, string $url = null)
    {
        $notificationData = [
            'title' => "🚨 BREAKING: {$title}",
            'message' => $message,
            'action_url' => $url,
            'data' => [
                'type' => 'breaking_news',
                'priority' => 'high',
            ],
            'web_buttons' => $url ? [
                [
                    'id' => 'read_now',
                    'text' => 'Read Now',
                    'icon' => 'https://via.placeholder.com/24x24/ff6b35/ffffff?text=📰',
                    'url' => $url,
                ]
            ] : [],
        ];

        return $this->sendToSegments(['All'], $notificationData);
    }
}

    public function __construct()
    {
        $this->appId = config('services.onesignal.app_id');
        $this->restApiKey = config('services.onesignal.rest_api_key');
    }

    /**
     * Send notification to specific players.
     */
    public function sendToPlayers(array $playerIds, array $notificationData)
    {
        $payload = [
            'app_id' => $this->appId,
            'include_player_ids' => $playerIds,
            'headings' => ['en' => $notificationData['title']],
            'contents' => ['en' => $notificationData['message']],
            'data' => $notificationData['data'] ?? [],
            'url' => $notificationData['action_url'] ?? null,
            'web_buttons' => $notificationData['web_buttons'] ?? [],
            'chrome_web_icon' => $notificationData['icon'] ?? null,
            'big_picture' => $notificationData['image_url'] ?? null,
            'send_after' => $notificationData['send_after'] ?? null,
        ];

        return $this->makeRequest('notifications', $payload);
    }

    /**
     * Send notification to segments.
     */
    public function sendToSegments(array $segments, array $notificationData)
    {
        $payload = [
            'app_id' => $this->appId,
            'included_segments' => $segments,
            'headings' => ['en' => $notificationData['title']],
            'contents' => ['en' => $notificationData['message']],
            'data' => $notificationData['data'] ?? [],
            'url' => $notificationData['action_url'] ?? null,
            'web_buttons' => $notificationData['web_buttons'] ?? [],
            'chrome_web_icon' => $notificationData['icon'] ?? null,
            'big_picture' => $notificationData['image_url'] ?? null,
            'send_after' => $notificationData['send_after'] ?? null,
        ];

        return $this->makeRequest('notifications', $payload);
    }

    /**
     * Send notification to all users.
     */
    public function sendToAll(array $notificationData)
    {
        return $this->sendToSegments(['All'], $notificationData);
    }

    /**
     * Send campaign notification.
     */
    public function sendCampaign(NotificationCampaign $campaign)
    {
        try {
            // Get target subscriptions
            $subscriptions = $this->getTargetSubscriptions($campaign);
            
            if ($subscriptions->isEmpty()) {
                Log::warning('No target subscriptions found for campaign', ['campaign_id' => $campaign->id]);
                return false;
            }

            $playerIds = $subscriptions->pluck('onesignal_player_id')->toArray();
            
            // Prepare notification data
            $notificationData = [
                'title' => $campaign->title,
                'message' => $campaign->message,
                'icon' => $campaign->icon,
                'image_url' => $campaign->image_url,
                'action_url' => $campaign->action_url,
                'data' => [
                    'campaign_id' => $campaign->id,
                    'type' => 'news_notification',
                    'utm_source' => 'onesignal',
                    'utm_medium' => 'push',
                    'utm_campaign' => 'news_' . $campaign->id,
                ],
                'web_buttons' => $campaign->action_text ? [
                    [
                        'id' => 'read_more',
                        'text' => $campaign->action_text,
                        'icon' => 'https://via.placeholder.com/24x24/ff6b35/ffffff?text=→',
                        'url' => $campaign->action_url,
                    ]
                ] : [],
            ];

            // Send notification
            $response = $this->sendToPlayers($playerIds, $notificationData);
            
            if ($response && isset($response['id'])) {
                // Update campaign
                $campaign->update([
                    'status' => 'sent',
                    'onesignal_campaign_id' => $response['id'],
                    'sent_count' => $response['recipients'] ?? count($playerIds),
                ]);

                // Create notification logs
                $this->createNotificationLogs($campaign, $subscriptions, $response['id']);
                
                Log::info('Campaign sent successfully', [
                    'campaign_id' => $campaign->id,
                    'onesignal_id' => $response['id'],
                    'recipients' => $response['recipients'] ?? count($playerIds),
                ]);

                return true;
            }

            return false;

        } catch (\Exception $e) {
            Log::error('Failed to send campaign', [
                'campaign_id' => $campaign->id,
                'error' => $e->getMessage(),
            ]);
            
            return false;
        }
    }

    /**
     * Get target subscriptions for a campaign.
     */
    protected function getTargetSubscriptions(NotificationCampaign $campaign)
    {
        $query = UserNotificationSubscription::active();

        // Filter by target audience
        if ($campaign->target_audience === 'registered') {
            $query->whereNotNull('user_id');
        } elseif ($campaign->target_audience === 'premium') {
            $query->whereHas('user', function ($q) {
                $q->where('is_premium', true);
            });
        }

        // Filter by categories
        if (!empty($campaign->target_categories)) {
            $query->where(function ($q) use ($campaign) {
                foreach ($campaign->target_categories as $category) {
                    $q->orWhereJsonContains('subscribed_categories', $category);
                }
            });
        }

        // Filter by locations
        if (!empty($campaign->target_locations)) {
            $query->where(function ($q) use ($campaign) {
                foreach ($campaign->target_locations as $location) {
                    $q->orWhereJsonContains('subscribed_locations', $location);
                }
            });
        }

        // Filter by interests
        if (!empty($campaign->target_interests)) {
            $query->where(function ($q) use ($campaign) {
                foreach ($campaign->target_interests as $interest) {
                    $q->orWhereJsonContains('subscribed_interests', $interest);
                }
            });
        }

        return $query->get();
    }

    /**
     * Create notification logs for sent notifications.
     */
    protected function createNotificationLogs(NotificationCampaign $campaign, $subscriptions, string $onesignalId)
    {
        $logs = [];
        
        foreach ($subscriptions as $subscription) {
            $logs[] = [
                'campaign_id' => $campaign->id,
                'subscription_id' => $subscription->id,
                'user_id' => $subscription->user_id,
                'onesignal_player_id' => $subscription->onesignal_player_id,
                'onesignal_notification_id' => $onesignalId,
                'status' => 'sent',
                'sent_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        NotificationLog::insert($logs);
    }

    /**
     * Get notification delivery status.
     */
    public function getNotificationStatus(string $notificationId)
    {
        return $this->makeRequest("notifications/{$notificationId}");
    }

    /**
     * Create or update player.
     */
    public function createPlayer(array $playerData)
    {
        $payload = array_merge([
            'app_id' => $this->appId,
        ], $playerData);

        return $this->makeRequest('players', $payload, 'POST');
    }

    /**
     * Update player.
     */
    public function updatePlayer(string $playerId, array $playerData)
    {
        return $this->makeRequest("players/{$playerId}", $playerData, 'PUT');
    }

    /**
     * Get player information.
     */
    public function getPlayer(string $playerId)
    {
        return $this->makeRequest("players/{$playerId}");
    }

    /**
     * Make HTTP request to OneSignal API.
     */
    public function makeRequest(string $endpoint, array $data = [], string $method = 'POST')
    {
        $url = "{$this->baseUrl}/{$endpoint}";
        
        // Always include app_id in the request body for OneSignal API
        $data['app_id'] = $this->appId;
        
        $response = Http::withHeaders([
            'Authorization' => "Basic {$this->restApiKey}",
            'Content-Type' => 'application/json',
        ])->$method($url, $data);

        if ($response->successful()) {
            return $response->json();
        }

        Log::error('OneSignal API request failed', [
            'endpoint' => $endpoint,
            'method' => $method,
            'status' => $response->status(),
            'response' => $response->body(),
            'request_data' => $data,
        ]);

        return null;
    }

    /**
     * Send breaking news notification.
     */
    public function sendBreakingNews(string $title, string $message, string $url = null)
    {
        $subscriptions = UserNotificationSubscription::active()
            ->where('breaking_news_enabled', true)
            ->get();

        if ($subscriptions->isEmpty()) {
            return false;
        }

        $playerIds = $subscriptions->pluck('onesignal_player_id')->toArray();

        $notificationData = [
            'title' => "🚨 BREAKING: {$title}",
            'message' => $message,
            'action_url' => $url,
            'data' => [
                'type' => 'breaking_news',
                'priority' => 'high',
            ],
            'web_buttons' => $url ? [
                [
                    'id' => 'read_now',
                    'text' => 'Read Now',
                    'icon' => 'https://via.placeholder.com/24x24/ff6b35/ffffff?text=📰',
                    'url' => $url,
                ]
            ] : [],
        ];

        return $this->sendToPlayers($playerIds, $notificationData);
    }

    /**
     * Send daily digest notification.
     */
    public function sendDailyDigest()
    {
        $subscriptions = UserNotificationSubscription::active()
            ->where('daily_digest_enabled', true)
            ->get();

        if ($subscriptions->isEmpty()) {
            return false;
        }

        $playerIds = $subscriptions->pluck('onesignal_player_id')->toArray();

        $notificationData = [
            'title' => '📰 Your Daily News Digest',
            'message' => 'Stay updated with today\'s top stories and breaking news.',
            'action_url' => route('news.chatbot'),
            'data' => [
                'type' => 'daily_digest',
                'priority' => 'normal',
            ],
            'web_buttons' => [
                [
                    'id' => 'read_digest',
                    'text' => 'Read Digest',
                    'icon' => 'https://via.placeholder.com/24x24/ff6b35/ffffff?text=📰',
                    'url' => route('news.chatbot'),
                ]
            ],
        ];

        return $this->sendToPlayers($playerIds, $notificationData);
    }

    /**
     * Get app information from OneSignal.
     */
    public function getAppInfo()
    {
        return $this->makeRequest("apps/{$this->appId}", [], 'GET');
    }

    /**
     * Get player count from OneSignal.
     */
    public function getPlayerCount()
    {
        $result = $this->makeRequest("players?app_id={$this->appId}&limit=1", [], 'GET');
        return $result['total_count'] ?? 0;
    }

    /**
     * Send breaking news notification to all users.
     */
    public function sendBreakingNewsToAll(string $title, string $message, string $url = null)
    {
        $notificationData = [
            'title' => "🚨 BREAKING: {$title}",
            'message' => $message,
            'action_url' => $url,
            'data' => [
                'type' => 'breaking_news',
                'priority' => 'high',
            ],
            'web_buttons' => $url ? [
                [
                    'id' => 'read_now',
                    'text' => 'Read Now',
                    'icon' => 'https://via.placeholder.com/24x24/ff6b35/ffffff?text=📰',
                    'url' => $url,
                ]
            ] : [],
        ];

        return $this->sendToSegments(['All'], $notificationData);
    }
}