<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Models\Election\Election;
use App\Models\Election\Party;
use App\Models\Election\Region;
use App\Models\Election\Poll;
use App\Models\Election\AIPrediction;
use App\Models\Election\ElectionResultRegional;
use App\Models\Election\SocialMention;

class ElectionAIService
{
    private $openaiApiKey;
    private $openaiModel;
    private $grokApiKey;
    private $grokModel;

    public function __construct()
    {
        $this->openaiApiKey = config('services.openai.api_key');
        $this->openaiModel = config('services.openai.model', 'gpt-4o-mini');
        $this->grokApiKey = config('services.grok.api_key');
        $this->grokModel = config('services.grok.model', 'grok-beta');
    }

    /**
     * Generate AI prediction for an election
     */
    public function generatePrediction(Election $election): array
    {
        try {
            // Gather all data sources
            $historicalData = $this->getHistoricalData($election);
            $pollData = $this->getPollData($election);
            $socialSentiment = $this->getSocialSentiment($election);
            $newsContext = $this->getNewsContext($election);

            // Build the prediction prompt
            $prompt = $this->buildPredictionPrompt($election, $historicalData, $pollData, $socialSentiment, $newsContext);

            // Call OpenAI for prediction
            $response = $this->callOpenAI($prompt);

            if (!$response) {
                return [
                    'success' => false,
                    'error' => 'Failed to generate prediction'
                ];
            }

            // Parse and store prediction
            $prediction = $this->parsePredictionResponse($response, $election);

            return [
                'success' => true,
                'prediction' => $prediction,
                'data_sources' => [
                    'historical_elections' => count($historicalData['elections'] ?? []),
                    'polls' => count($pollData),
                    'social_mentions' => $socialSentiment['total_mentions'] ?? 0,
                    'news_articles' => count($newsContext),
                ]
            ];

        } catch (\Exception $e) {
            Log::error('Election AI prediction error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Get historical election data for analysis
     */
    private function getHistoricalData(Election $election): array
    {
        $parties = Party::whereIn('short_name', ['NPP', 'NDC'])->get();
        
        // Get past elections of the same type
        $pastElections = Election::where('type', $election->type)
            ->where('status', 'completed')
            ->where('year', '<', $election->year)
            ->orderByDesc('year')
            ->limit(5)
            ->get();

        $regionalTrends = [];
        foreach (Region::all() as $region) {
            $regionResults = ElectionResultRegional::where('region_id', $region->id)
                ->whereIn('election_id', $pastElections->pluck('id'))
                ->with(['election', 'party'])
                ->orderBy('election_id')
                ->get()
                ->groupBy('election_id');
            
            $regionalTrends[$region->name] = $regionResults->map(function ($results, $electionId) {
                $npp = $results->where('party.short_name', 'NPP')->first();
                $ndc = $results->where('party.short_name', 'NDC')->first();
                return [
                    'npp_pct' => $npp?->percentage ?? 0,
                    'ndc_pct' => $ndc?->percentage ?? 0,
                    'winner' => $npp?->is_region_winner ? 'NPP' : ($ndc?->is_region_winner ? 'NDC' : 'Other'),
                ];
            })->toArray();
        }

        return [
            'elections' => $pastElections->toArray(),
            'regional_trends' => $regionalTrends,
            'parties' => $parties->toArray(),
        ];
    }

    /**
     * Get poll data for the election
     */
    private function getPollData(Election $election): array
    {
        return Poll::where('election_id', $election->id)
            ->with(['results.party', 'results.candidate'])
            ->orderByDesc('date_conducted')
            ->limit(10)
            ->get()
            ->toArray();
    }

    /**
     * Get social media sentiment analysis
     */
    private function getSocialSentiment(Election $election): array
    {
        $mentions = SocialMention::where('election_id', $election->id)
            ->where('captured_at', '>=', now()->subDays(30))
            ->get();

        if ($mentions->isEmpty()) {
            return [
                'total_mentions' => 0,
                'sentiment_breakdown' => [],
                'party_sentiment' => [],
            ];
        }

        $sentimentBreakdown = [
            'positive' => $mentions->where('sentiment', 'positive')->count(),
            'negative' => $mentions->where('sentiment', 'negative')->count(),
            'neutral' => $mentions->where('sentiment', 'neutral')->count(),
        ];

        $partySentiment = [];
        foreach (Party::whereIn('short_name', ['NPP', 'NDC'])->get() as $party) {
            $partyMentions = $mentions->where('party_id', $party->id);
            $partySentiment[$party->short_name] = [
                'total' => $partyMentions->count(),
                'positive' => $partyMentions->where('sentiment', 'positive')->count(),
                'negative' => $partyMentions->where('sentiment', 'negative')->count(),
                'avg_engagement' => $partyMentions->avg('engagement_score') ?? 0,
            ];
        }

        return [
            'total_mentions' => $mentions->count(),
            'sentiment_breakdown' => $sentimentBreakdown,
            'party_sentiment' => $partySentiment,
        ];
    }

    /**
     * Get relevant news context from Mews crawler
     */
    private function getNewsContext(Election $election): array
    {
        try {
            $keywords = ['election', 'NPP', 'NDC', 'presidential', 'parliament', 'vote', 'Ghana politics'];
            
            $news = DB::connection('crawler')
                ->table('vw_crawler_mews')
                ->where('sub_category', 'like', '%politic%')
                ->orWhere('sub_category', 'like', '%election%')
                ->where('date_added', '>=', now()->subDays(30))
                ->orderByDesc('date_added')
                ->limit(20)
                ->get(['post_title', 'post_content', 'date_added', 'post_source']);

            return $news->toArray();
        } catch (\Exception $e) {
            Log::warning('Failed to fetch news context: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Build the prediction prompt
     */
    private function buildPredictionPrompt(Election $election, array $historicalData, array $pollData, array $socialSentiment, array $newsContext): string
    {
        $prompt = "You are an expert political analyst specializing in Ghana elections. Analyze the following data and provide an objective, data-driven prediction for the {$election->name}.\n\n";
        
        $prompt .= "IMPORTANT GUIDELINES:\n";
        $prompt .= "- Be strictly objective and non-partisan\n";
        $prompt .= "- Base all claims on the provided data\n";
        $prompt .= "- Cite specific data points for each claim\n";
        $prompt .= "- Acknowledge uncertainty and limitations\n";
        $prompt .= "- Do not make personal judgments about candidates\n\n";

        // Historical data
        $prompt .= "HISTORICAL ELECTION DATA:\n";
        foreach ($historicalData['regional_trends'] as $region => $trends) {
            $prompt .= "- {$region}: ";
            $avgNpp = collect($trends)->avg('npp_pct');
            $avgNdc = collect($trends)->avg('ndc_pct');
            $prompt .= "NPP avg: " . round($avgNpp, 1) . "%, NDC avg: " . round($avgNdc, 1) . "%\n";
        }

        // Poll data
        if (!empty($pollData)) {
            $prompt .= "\nRECENT POLLS:\n";
            foreach (array_slice($pollData, 0, 5) as $poll) {
                $prompt .= "- {$poll['pollster_name']} ({$poll['date_conducted']}): ";
                foreach ($poll['results'] ?? [] as $result) {
                    $partyName = $result['party']['short_name'] ?? 'Unknown';
                    $prompt .= "{$partyName}: {$result['percentage']}% ";
                }
                if ($poll['margin_of_error']) {
                    $prompt .= "(MOE: ±{$poll['margin_of_error']}%)";
                }
                $prompt .= "\n";
            }
        }

        // Social sentiment
        if ($socialSentiment['total_mentions'] > 0) {
            $prompt .= "\nSOCIAL MEDIA SENTIMENT (last 30 days):\n";
            foreach ($socialSentiment['party_sentiment'] as $party => $data) {
                $positiveRate = $data['total'] > 0 ? round(($data['positive'] / $data['total']) * 100) : 0;
                $prompt .= "- {$party}: {$data['total']} mentions, {$positiveRate}% positive\n";
            }
        }

        // News context (summarized)
        if (!empty($newsContext)) {
            $prompt .= "\nRECENT NEWS THEMES:\n";
            $prompt .= "- " . count($newsContext) . " political articles analyzed from the past 30 days\n";
        }

        $prompt .= "\nPlease provide your analysis in the following JSON format:\n";
        $prompt .= "{\n";
        $prompt .= '  "predictions": [' . "\n";
        $prompt .= '    {"party": "NPP", "win_probability": <0-100>, "vote_share": <0-100>, "confidence_lower": <number>, "confidence_upper": <number>},' . "\n";
        $prompt .= '    {"party": "NDC", "win_probability": <0-100>, "vote_share": <0-100>, "confidence_lower": <number>, "confidence_upper": <number>}' . "\n";
        $prompt .= "  ],\n";
        $prompt .= '  "qualitative_analysis": "<2-3 paragraph analysis explaining the prediction>",';
        $prompt .= "\n";
        $prompt .= '  "key_factors": ["<factor 1>", "<factor 2>", "<factor 3>"],';
        $prompt .= "\n";
        $prompt .= '  "risks_and_caveats": "<paragraph about uncertainties and limitations>",';
        $prompt .= "\n";
        $prompt .= '  "regional_outlook": {"<region>": "<brief outlook>", ...}';
        $prompt .= "\n}\n";

        return $prompt;
    }

    /**
     * Call OpenAI API
     */
    private function callOpenAI(string $prompt): ?array
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->openaiApiKey,
                'Content-Type' => 'application/json',
            ])->timeout(60)->post('https://api.openai.com/v1/chat/completions', [
                'model' => $this->openaiModel,
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are an expert political analyst. Always respond with valid JSON only, no markdown formatting.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'max_tokens' => 2000,
                'temperature' => 0.3, // Lower temperature for more consistent predictions
            ]);

            if ($response->successful()) {
                $content = $response->json('choices.0.message.content');
                // Clean up any markdown formatting
                $content = preg_replace('/```json\s*/', '', $content);
                $content = preg_replace('/```\s*/', '', $content);
                return json_decode($content, true);
            }

            Log::error('OpenAI API error', [
                'status' => $response->status(),
                'response' => $response->body()
            ]);
            return null;

        } catch (\Exception $e) {
            Log::error('OpenAI API exception: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Parse and store prediction response
     */
    private function parsePredictionResponse(array $response, Election $election): array
    {
        $predictions = [];
        
        foreach ($response['predictions'] ?? [] as $prediction) {
            $party = Party::where('short_name', $prediction['party'])->first();
            
            if ($party) {
                $aiPrediction = AIPrediction::updateOrCreate(
                    [
                        'election_id' => $election->id,
                        'party_id' => $party->id,
                    ],
                    [
                        'win_probability' => $prediction['win_probability'] ?? null,
                        'vote_share_prediction' => $prediction['vote_share'] ?? null,
                        'lower_confidence' => $prediction['confidence_lower'] ?? null,
                        'upper_confidence' => $prediction['confidence_upper'] ?? null,
                        'qualitative_analysis' => $response['qualitative_analysis'] ?? null,
                        'key_factors' => json_encode($response['key_factors'] ?? []),
                        'risks_and_caveats' => $response['risks_and_caveats'] ?? null,
                        'data_sources' => [
                            'historical' => true,
                            'polls' => true,
                            'social' => true,
                            'news' => true,
                        ],
                        'model_version' => $this->openaiModel,
                        'generated_at' => now(),
                    ]
                );
                
                $predictions[] = $aiPrediction;
            }
        }

        return [
            'predictions' => $predictions,
            'analysis' => $response['qualitative_analysis'] ?? '',
            'key_factors' => $response['key_factors'] ?? [],
            'risks' => $response['risks_and_caveats'] ?? '',
            'regional_outlook' => $response['regional_outlook'] ?? [],
        ];
    }

    /**
     * Answer election-related questions using AI
     */
    public function answerElectionQuestion(string $question, ?int $electionId = null): array
    {
        try {
            // Gather context
            $context = $this->buildQuestionContext($question, $electionId);
            
            $prompt = "You are Mews Election Intelligence, an AI assistant specializing in Ghana elections and politics.\n\n";
            $prompt .= "GUIDELINES:\n";
            $prompt .= "- Be objective and non-partisan\n";
            $prompt .= "- Base answers on the provided data\n";
            $prompt .= "- Cite specific sources when making claims\n";
            $prompt .= "- Acknowledge when you don't have enough data\n";
            $prompt .= "- Do not make personal judgments about individuals\n\n";
            $prompt .= "CONTEXT DATA:\n{$context}\n\n";
            $prompt .= "USER QUESTION: {$question}\n\n";
            $prompt .= "Provide a helpful, factual response:";

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->openaiApiKey,
                'Content-Type' => 'application/json',
            ])->timeout(30)->post('https://api.openai.com/v1/chat/completions', [
                'model' => $this->openaiModel,
                'messages' => [
                    ['role' => 'system', 'content' => 'You are an expert on Ghana elections and politics. Be factual and objective.'],
                    ['role' => 'user', 'content' => $prompt]
                ],
                'max_tokens' => 1000,
                'temperature' => 0.5,
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'response' => $response->json('choices.0.message.content'),
                    'sources_used' => ['historical_data', 'polls', 'news'],
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to get AI response'
            ];

        } catch (\Exception $e) {
            Log::error('Election question error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Build context for answering questions
     */
    private function buildQuestionContext(string $question, ?int $electionId): string
    {
        $context = "";

        // Get recent election data
        $elections = Election::completed()
            ->presidential()
            ->orderByDesc('year')
            ->limit(5)
            ->get();

        $context .= "RECENT PRESIDENTIAL ELECTIONS:\n";
        foreach ($elections as $election) {
            $context .= "- {$election->year}: {$election->name}\n";
            
            // Get regional winners summary
            $nppWins = ElectionResultRegional::where('election_id', $election->id)
                ->whereHas('party', fn($q) => $q->where('short_name', 'NPP'))
                ->where('is_region_winner', true)
                ->count();
            
            $ndcWins = ElectionResultRegional::where('election_id', $election->id)
                ->whereHas('party', fn($q) => $q->where('short_name', 'NDC'))
                ->where('is_region_winner', true)
                ->count();
            
            $context .= "  NPP won {$nppWins} regions, NDC won {$ndcWins} regions\n";
        }

        // Get recent polls if specific election
        if ($electionId) {
            $polls = Poll::where('election_id', $electionId)
                ->with('results.party')
                ->orderByDesc('date_conducted')
                ->limit(5)
                ->get();

            if ($polls->isNotEmpty()) {
                $context .= "\nRECENT POLLS:\n";
                foreach ($polls as $poll) {
                    $context .= "- {$poll->pollster_name} ({$poll->date_conducted->format('M j, Y')}): ";
                    foreach ($poll->results as $result) {
                        $context .= "{$result->party->short_name}: {$result->percentage}% ";
                    }
                    $context .= "\n";
                }
            }
        }

        return $context;
    }

    /**
     * Get social sentiment using Grok API (for X/Twitter analysis)
     */
    public function getGrokSentiment(string $topic): ?array
    {
        if (!$this->grokApiKey) {
            return null;
        }

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->grokApiKey,
                'Content-Type' => 'application/json',
            ])->timeout(30)->post(config('services.grok.base_url') . '/chat/completions', [
                'model' => $this->grokModel,
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are an AI that analyzes social media sentiment on X/Twitter. Provide objective analysis.'
                    ],
                    [
                        'role' => 'user',
                        'content' => "Analyze the current sentiment on X/Twitter about: {$topic}. Focus on Ghana political context. Provide sentiment breakdown (positive/negative/neutral percentages) and key themes being discussed."
                    ]
                ],
                'max_tokens' => 500,
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'analysis' => $response->json('choices.0.message.content'),
                    'source' => 'grok',
                ];
            }

            return null;

        } catch (\Exception $e) {
            Log::error('Grok API error: ' . $e->getMessage());
            return null;
        }
    }
}
