<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Services\ImageService;

class UserProfile extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'first_name',
        'last_name',
        'phone',
        'date_of_birth',
        'gender',
        'location',
        'bio',
        'avatar',
        'interests',
        'preferred_sources',
        'language_preference',
        'email_notifications',
        'push_notifications',
        'last_active_at',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'interests' => 'array',
        'preferred_sources' => 'array',
        'email_notifications' => 'boolean',
        'push_notifications' => 'boolean',
        'last_active_at' => 'datetime',
    ];

    /**
     * Get the user that owns the profile.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the user's full name.
     */
    public function getFullNameAttribute()
    {
        return trim($this->first_name . ' ' . $this->last_name);
    }

    /**
     * Get the user's age.
     */
    public function getAgeAttribute()
    {
        if (!$this->date_of_birth) {
            return null;
        }
        
        return $this->date_of_birth->age;
    }

    /**
     * Get the user's avatar URL.
     */
    public function getAvatarUrlAttribute()
    {
        if ($this->avatar) {
            return ImageService::getImageUrl($this->avatar, 'public');
        }
        
        // Generate a default avatar based on user's name
        $fullName = trim(($this->first_name ?? '') . ' ' . ($this->last_name ?? ''));
        return ImageService::getDefaultAvatarUrl($fullName);
    }

    /**
     * Update last active timestamp.
     */
    public function updateLastActive()
    {
        $this->update(['last_active_at' => now()]);
    }
}