<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class UserNotificationSubscription extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'onesignal_player_id',
        'device_type',
        'browser',
        'os',
        'country',
        'timezone',
        'subscribed_categories',
        'subscribed_locations',
        'subscribed_interests',
        'breaking_news_enabled',
        'daily_digest_enabled',
        'weekly_summary_enabled',
        'marketing_enabled',
        'quiet_hours_start',
        'quiet_hours_end',
        'preferred_days',
        'preferred_times',
        'is_active',
        'last_activity_at',
        'subscribed_at',
    ];

    protected $casts = [
        'subscribed_categories' => 'array',
        'subscribed_locations' => 'array',
        'subscribed_interests' => 'array',
        'preferred_days' => 'array',
        'preferred_times' => 'array',
        'breaking_news_enabled' => 'boolean',
        'daily_digest_enabled' => 'boolean',
        'weekly_summary_enabled' => 'boolean',
        'marketing_enabled' => 'boolean',
        'is_active' => 'boolean',
        'last_activity_at' => 'datetime',
        'subscribed_at' => 'datetime',
    ];

    /**
     * Get the user that owns the subscription.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the notification logs for this subscription.
     */
    public function logs(): HasMany
    {
        return $this->hasMany(NotificationLog::class, 'subscription_id');
    }

    /**
     * Scope for active subscriptions.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for subscriptions by category.
     */
    public function scopeByCategory($query, string $category)
    {
        return $query->whereJsonContains('subscribed_categories', $category);
    }

    /**
     * Scope for subscriptions by location.
     */
    public function scopeByLocation($query, string $location)
    {
        return $query->whereJsonContains('subscribed_locations', $location);
    }

    /**
     * Scope for subscriptions by interest.
     */
    public function scopeByInterest($query, string $interest)
    {
        return $query->whereJsonContains('subscribed_interests', $interest);
    }

    /**
     * Check if subscription is in quiet hours.
     */
    public function isInQuietHours()
    {
        if (!$this->quiet_hours_start || !$this->quiet_hours_end) {
            return false;
        }

        $now = now()->setTimezone($this->timezone ?? 'UTC');
        $currentTime = $now->format('H:i:s');
        
        $start = $this->quiet_hours_start;
        $end = $this->quiet_hours_end;

        // Handle overnight quiet hours (e.g., 22:00 to 08:00)
        if ($start > $end) {
            return $currentTime >= $start || $currentTime <= $end;
        }

        return $currentTime >= $start && $currentTime <= $end;
    }

    /**
     * Check if subscription allows notifications at current time.
     */
    public function canReceiveNotificationAt($time = null)
    {
        $time = $time ?? now();
        
        // Check quiet hours
        if ($this->isInQuietHours()) {
            return false;
        }

        // Check preferred days
        if (!empty($this->preferred_days)) {
            $dayName = strtolower($time->format('l'));
            if (!in_array($dayName, $this->preferred_days)) {
                return false;
            }
        }

        // Check preferred times (within 30 minutes)
        if (!empty($this->preferred_times)) {
            $currentTime = $time->format('H:i');
            $canReceive = false;
            
            foreach ($this->preferred_times as $preferredTime) {
                $preferred = \Carbon\Carbon::createFromFormat('H:i', $preferredTime);
                $current = \Carbon\Carbon::createFromFormat('H:i', $currentTime);
                
                if ($current->diffInMinutes($preferred) <= 30) {
                    $canReceive = true;
                    break;
                }
            }
            
            if (!$canReceive) {
                return false;
            }
        }

        return true;
    }

    /**
     * Check if subscription is interested in category.
     */
    public function isInterestedInCategory(string $category)
    {
        if (empty($this->subscribed_categories)) {
            return true; // No specific preferences
        }
        
        return in_array($category, $this->subscribed_categories);
    }

    /**
     * Check if subscription is interested in location.
     */
    public function isInterestedInLocation(string $location)
    {
        if (empty($this->subscribed_locations)) {
            return true; // No specific preferences
        }
        
        return in_array($location, $this->subscribed_locations);
    }

    /**
     * Check if subscription is interested in interest.
     */
    public function isInterestedInInterest(string $interest)
    {
        if (empty($this->subscribed_interests)) {
            return true; // No specific preferences
        }
        
        return in_array($interest, $this->subscribed_interests);
    }

    /**
     * Update last activity timestamp.
     */
    public function updateActivity()
    {
        $this->update(['last_activity_at' => now()]);
    }
}