<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class QueryLog extends Model
{
    use HasFactory;

    public $timestamps = false; // Table only has created_at, not updated_at

    protected $fillable = [
        'user_id',
        'session_id',
        'query',
        'intent',
        'response_data',
        'response_time_ms',
        'tokens_used',
        'model_used',
        'was_successful',
        'error_message',
        'user_ip',
        'user_agent',
        'referrer',
        'context_data',
    ];

    protected $casts = [
        'response_data' => 'array',
        'was_successful' => 'boolean',
        'context_data' => 'array',
        'created_at' => 'datetime',
    ];

    /**
     * Get the user that made the query.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Scope for successful queries.
     */
    public function scopeSuccessful($query)
    {
        return $query->where('was_successful', true);
    }

    /**
     * Scope for failed queries.
     */
    public function scopeFailed($query)
    {
        return $query->where('was_successful', false);
    }

    /**
     * Scope for queries by intent.
     */
    public function scopeByIntent($query, $intent)
    {
        return $query->where('intent', $intent);
    }

    /**
     * Scope for recent queries.
     */
    public function scopeRecent($query, $days = 7)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    /**
     * Scope for anonymous queries.
     */
    public function scopeAnonymous($query)
    {
        return $query->whereNull('user_id');
    }

    /**
     * Scope for authenticated user queries.
     */
    public function scopeAuthenticated($query)
    {
        return $query->whereNotNull('user_id');
    }

    /**
     * Get the response time in a human-readable format.
     */
    public function getResponseTimeFormattedAttribute()
    {
        if (!$this->response_time_ms) {
            return 'Unknown';
        }

        if ($this->response_time_ms < 1000) {
            return $this->response_time_ms . 'ms';
        }

        return round($this->response_time_ms / 1000, 2) . 's';
    }

    /**
     * Get the query intent display name.
     */
    public function getIntentDisplayAttribute()
    {
        $intents = [
            'news_search' => 'News Search',
            'general_chat' => 'General Chat',
            'article_request' => 'Article Request',
            'category_filter' => 'Category Filter',
            'trending_topics' => 'Trending Topics',
            'weather' => 'Weather',
            'sports' => 'Sports',
            'politics' => 'Politics',
            'business' => 'Business',
            'technology' => 'Technology',
        ];

        return $intents[$this->intent] ?? ucfirst(str_replace('_', ' ', $this->intent));
    }
}