<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class NotificationCampaign extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'message',
        'icon',
        'image_url',
        'action_url',
        'action_text',
        'target_categories',
        'target_locations',
        'target_interests',
        'target_audience',
        'scheduled_at',
        'expires_at',
        'is_immediate',
        'status',
        'is_active',
        'sent_count',
        'delivered_count',
        'clicked_count',
        'converted_count',
        'onesignal_segment',
        'onesignal_campaign_id',
    ];

    protected $casts = [
        'target_categories' => 'array',
        'target_locations' => 'array',
        'target_interests' => 'array',
        'scheduled_at' => 'datetime',
        'expires_at' => 'datetime',
        'is_immediate' => 'boolean',
        'is_active' => 'boolean',
    ];

    /**
     * Get the notification logs for this campaign.
     */
    public function logs(): HasMany
    {
        return $this->hasMany(NotificationLog::class, 'campaign_id');
    }

    /**
     * Scope for active campaigns.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for scheduled campaigns.
     */
    public function scopeScheduled($query)
    {
        return $query->where('status', 'scheduled');
    }

    /**
     * Scope for sent campaigns.
     */
    public function scopeSent($query)
    {
        return $query->where('status', 'sent');
    }

    /**
     * Get the delivery rate.
     */
    public function getDeliveryRateAttribute()
    {
        if ($this->sent_count == 0) {
            return 0;
        }
        return round(($this->delivered_count / $this->sent_count) * 100, 2);
    }

    /**
     * Get the click rate.
     */
    public function getClickRateAttribute()
    {
        if ($this->delivered_count == 0) {
            return 0;
        }
        return round(($this->clicked_count / $this->delivered_count) * 100, 2);
    }

    /**
     * Get the conversion rate.
     */
    public function getConversionRateAttribute()
    {
        if ($this->clicked_count == 0) {
            return 0;
        }
        return round(($this->converted_count / $this->clicked_count) * 100, 2);
    }

    /**
     * Check if campaign is ready to send.
     */
    public function isReadyToSend()
    {
        return $this->status === 'scheduled' && 
               $this->is_active && 
               $this->scheduled_at <= now();
    }

    /**
     * Check if campaign targets specific categories.
     */
    public function targetsCategories(array $categories)
    {
        if (empty($this->target_categories)) {
            return true; // No specific targeting
        }
        
        return !empty(array_intersect($this->target_categories, $categories));
    }

    /**
     * Check if campaign targets specific locations.
     */
    public function targetsLocations(array $locations)
    {
        if (empty($this->target_locations)) {
            return true; // No specific targeting
        }
        
        return !empty(array_intersect($this->target_locations, $locations));
    }

    /**
     * Check if campaign targets specific interests.
     */
    public function targetsInterests(array $interests)
    {
        if (empty($this->target_interests)) {
            return true; // No specific targeting
        }
        
        return !empty(array_intersect($this->target_interests, $interests));
    }
}