<?php

namespace App\Models\Election;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Candidate extends Model
{
    protected $fillable = [
        'name',
        'party_id',
        'photo_url',
        'biography',
        'date_of_birth',
        'position',
        'is_active',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'is_active' => 'boolean',
    ];

    public function party(): BelongsTo
    {
        return $this->belongsTo(Party::class);
    }

    public function electionCandidacies(): HasMany
    {
        return $this->hasMany(ElectionCandidate::class);
    }

    public function nationalResults(): HasMany
    {
        return $this->hasMany(ElectionResultNational::class);
    }

    public function regionalResults(): HasMany
    {
        return $this->hasMany(ElectionResultRegional::class);
    }

    public function constituencyResults(): HasMany
    {
        return $this->hasMany(ElectionResultConstituency::class);
    }

    public function pollResults(): HasMany
    {
        return $this->hasMany(PollResult::class);
    }

    public function predictions(): HasMany
    {
        return $this->hasMany(AIPrediction::class);
    }

    /**
     * Get elections this candidate participated in
     */
    public function getElections()
    {
        return Election::whereHas('candidates', function ($query) {
            $query->where('candidate_id', $this->id);
        })->get();
    }

    /**
     * Get the candidate's party color (for charts)
     */
    public function getPartyColorAttribute(): string
    {
        return $this->party?->color ?? '#808080';
    }
}
