<?php

namespace App\Models\Election;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class AIPrediction extends Model
{
    protected $table = 'ai_predictions';

    protected $fillable = [
        'election_id',
        'candidate_id',
        'party_id',
        'win_probability',
        'vote_share_prediction',
        'lower_confidence',
        'upper_confidence',
        'qualitative_analysis',
        'key_factors',
        'risks_and_caveats',
        'data_sources',
        'model_version',
        'generated_at',
    ];

    protected $casts = [
        'win_probability' => 'decimal:2',
        'vote_share_prediction' => 'decimal:2',
        'lower_confidence' => 'decimal:2',
        'upper_confidence' => 'decimal:2',
        'data_sources' => 'json',
        'generated_at' => 'datetime',
    ];

    public function election(): BelongsTo
    {
        return $this->belongsTo(Election::class);
    }

    public function candidate(): BelongsTo
    {
        return $this->belongsTo(Candidate::class);
    }

    public function party(): BelongsTo
    {
        return $this->belongsTo(Party::class);
    }

    /**
     * Get confidence interval as string
     */
    public function getConfidenceRangeAttribute(): ?string
    {
        if ($this->lower_confidence && $this->upper_confidence) {
            return "{$this->lower_confidence}% - {$this->upper_confidence}%";
        }
        return null;
    }

    /**
     * Get key factors as array
     */
    public function getKeyFactorsArrayAttribute(): array
    {
        if (is_string($this->key_factors)) {
            return json_decode($this->key_factors, true) ?? [];
        }
        return $this->key_factors ?? [];
    }

    /**
     * Get probability classification
     */
    public function getProbabilityClassAttribute(): string
    {
        if ($this->win_probability >= 75) return 'likely';
        if ($this->win_probability >= 55) return 'leaning';
        if ($this->win_probability >= 45) return 'tossup';
        if ($this->win_probability >= 25) return 'leaning_against';
        return 'unlikely';
    }

    /**
     * Get probability display label
     */
    public function getProbabilityLabelAttribute(): string
    {
        return match($this->probability_class) {
            'likely' => 'Likely Win',
            'leaning' => 'Leaning',
            'tossup' => 'Toss-up',
            'leaning_against' => 'Leaning Against',
            'unlikely' => 'Unlikely Win',
            default => 'Unknown',
        };
    }
}
