<?php

namespace App\Http\Controllers;

use App\Models\ReadingHistory;
use App\Models\UserBookmark;
use App\Models\UserPreference;
use App\Models\UserProfile;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show user dashboard/profile.
     */
    public function dashboard()
    {
        $user = Auth::user();
        $user->load(['profile', 'preferences', 'readingHistory']);

        // Get recent reading history (limit to 2 most recent)
        $recentHistory = $user->readingHistory()
            ->orderBy('read_at', 'desc')
            ->limit(2)
            ->get();

        // Get reading statistics
        $stats = [
            'total_articles_read' => $user->readingHistory()->count(),
            'articles_this_week' => $user->readingHistory()->where('read_at', '>=', now()->subDays(7))->count(),
        ];

        return view('user.dashboard', compact('user', 'recentHistory', 'stats'));
    }

    /**
     * Show user profile edit form.
     */
    public function editProfile()
    {
        $user = Auth::user();
        $user->load('profile');

        return view('user.profile-edit', compact('user'));
    }

    /**
     * Update user profile.
     */
    public function updateProfile(Request $request)
    {
        $user = Auth::user();
        

        $validator = Validator::make($request->all(), [
            'name' => ['required', 'string', 'max:255'],
            'first_name' => ['nullable', 'string', 'max:255'],
            'last_name' => ['nullable', 'string', 'max:255'],
            'phone' => ['nullable', 'string', 'max:20'],
            'location' => ['nullable', 'string', 'max:255'],
            'bio' => ['nullable', 'string', 'max:1000'],
            'interests' => ['nullable', 'array'],
            'interests.*' => ['string', 'max:255'],
            'language_preference' => ['nullable', 'string', 'in:en,twi,ga,ewe'],
            'avatar' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif,webp', 'max:2048'],
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Update user basic info
        $user->update([
            'name' => $request->name,
        ]);

        // Update or create profile
        $profileData = [
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'phone' => $request->phone,
            'location' => $request->location,
            'bio' => $request->bio,
            'interests' => $request->interests ?? [],
            'language_preference' => $request->language_preference ?? 'en',
        ];

        // Handle avatar upload or removal
        if ($request->hasFile('avatar')) {
            // Delete old avatar if exists
            if ($user->profile && $user->profile->avatar) {
                \App\Services\ImageService::deleteImage($user->profile->avatar, 'public');
            }

            $path = \App\Services\ImageService::storeUploadedFile($request->file('avatar'), 'avatars', 'public');
            $profileData['avatar'] = $path;
        } elseif ($request->has('remove_avatar') && $request->remove_avatar == '1') {
            // Remove avatar if requested
            if ($user->profile && $user->profile->avatar) {
                \App\Services\ImageService::deleteImage($user->profile->avatar, 'public');
                $profileData['avatar'] = null;
            }
        }

        if ($user->profile) {
            $user->profile->update($profileData);
        } else {
            $profileData['user_id'] = $user->id;
            $user->profile()->create($profileData);
        }

        return redirect()->route('user.dashboard')
            ->with('success', 'Your profile has been updated successfully.');
    }

    /**
     * Show user preferences edit form.
     */
    public function editPreferences()
    {
        $user = Auth::user();
        $user->load('preferences');

        return view('user.preferences', compact('user'));
    }

    /**
     * Update user preferences.
     */
    public function updatePreferences(Request $request)
    {
        $user = Auth::user();
        

        // Handle AJAX theme-only updates
        if ($request->ajax() && $request->has('theme')) {
            $validator = Validator::make($request->all(), [
                'theme' => ['required', 'string', 'in:light,dark,auto'],
            ]);

            if ($validator->fails()) {
                return response()->json(['error' => 'Invalid theme'], 400);
            }

            if ($user->preferences) {
                $user->preferences->update(['theme' => $request->theme]);
            } else {
                $user->preferences()->create([
                    'user_id' => $user->id,
                    'theme' => $request->theme,
                    ...UserPreference::getDefaults()
                ]);
            }

            return response()->json(['success' => true, 'theme' => $request->theme]);
        }

        // Handle full form updates
        $validator = Validator::make($request->all(), [
            'theme' => ['required', 'string', 'in:light,dark,auto'],
            'font_size' => ['required', 'string', 'in:small,medium,large'],
            'auto_play_videos' => ['boolean'],
            'show_images' => ['boolean'],
            'news_display_mode' => ['required', 'string', 'in:cards,list,compact'],
            'articles_per_page' => ['required', 'integer', 'min:5', 'max:50'],
            'ai_suggestions' => ['boolean'],
            'trending_alerts' => ['boolean'],
            'timezone' => ['required', 'string', 'max:255'],
            'interests' => ['nullable', 'array'],
            'interests.*' => ['string', 'in:Politics,Business,Entertainment,Sports,International,General,Local,Technology'],
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $preferencesData = [
            'theme' => $request->theme,
            'font_size' => $request->font_size,
            'auto_play_videos' => $request->has('auto_play_videos'),
            'show_images' => $request->has('show_images'),
            'news_display_mode' => $request->news_display_mode,
            'articles_per_page' => $request->articles_per_page,
            'ai_suggestions' => $request->has('ai_suggestions'),
            'trending_alerts' => $request->has('trending_alerts'),
            'timezone' => $request->timezone,
        ];

        if ($user->preferences) {
            $user->preferences->update($preferencesData);
        } else {
            $preferencesData['user_id'] = $user->id;
            $user->preferences()->create($preferencesData);
        }

        // Update user interests if provided
        if ($request->has('interests')) {
            if ($user->profile) {
                $user->profile->update(['interests' => $request->interests ?? []]);
            } else {
                $user->profile()->create([
                    'user_id' => $user->id,
                    'interests' => $request->interests ?? [],
                ]);
            }
        }

        return redirect()->route('user.dashboard')
            ->with('success', 'Your preferences have been updated successfully.');
    }

    /**
     * Show user's reading history.
     */
    public function readingHistory()
    {
        $user = Auth::user();
        
        $history = $user->readingHistory()
            ->orderBy('read_at', 'desc')
            ->paginate(20);

        return view('user.reading-history', compact('user', 'history'));
    }

    /**
     * Show user's bookmarks.
     */
    public function bookmarks()
    {
        $user = Auth::user();
        
        $bookmarks = $user->bookmarks()
            ->orderBy('bookmarked_at', 'desc')
            ->paginate(20);

        return view('user.bookmarks', compact('user', 'bookmarks'));
    }

    /**
     * Add a bookmark.
     */
    public function addBookmark(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'article_id' => ['required', 'string'],
            'article_title' => ['required', 'string'],
            'article_category' => ['nullable', 'string'],
            'article_source' => ['nullable', 'string'],
            'notes' => ['nullable', 'string', 'max:1000'],
            'tags' => ['nullable', 'array'],
            'tags.*' => ['string', 'max:255'],
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => 'Invalid data'], 400);
        }

        $user = Auth::user();

        // Check if already bookmarked
        $existingBookmark = $user->bookmarks()
            ->where('article_id', $request->article_id)
            ->first();

        if ($existingBookmark) {
            return response()->json(['error' => 'Article already bookmarked'], 400);
        }

        $bookmark = $user->bookmarks()->create([
            'article_id' => $request->article_id,
            'article_title' => $request->article_title,
            'article_category' => $request->article_category,
            'article_source' => $request->article_source,
            'notes' => $request->notes,
            'tags' => $request->tags ?? [],
            'bookmarked_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'bookmark' => $bookmark
        ]);
    }

    /**
     * Remove a bookmark.
     */
    public function removeBookmark(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'article_id' => ['required', 'string'],
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => 'Invalid data'], 400);
        }

        $user = Auth::user();

        $bookmark = $user->bookmarks()
            ->where('article_id', $request->article_id)
            ->first();

        if (!$bookmark) {
            return response()->json(['error' => 'Bookmark not found'], 404);
        }

        $bookmark->delete();

        return response()->json(['success' => true]);
    }

}