<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Models\QueryLog;
use App\Models\ChatSession;
use App\Models\Election\RegionCensus;
use App\Services\OpenAIWebService;
use Illuminate\Support\Facades\Storage;

class NewsChatbotController extends Controller
{
    private $openaiApiKey;
    private $openaiModel;
    private $aiSettings;
    private $openAIWebService;

    public function __construct()
    {
        $this->openaiApiKey = config('services.openai.api_key');
        
        // Get AI settings from database, fallback to config if not found
        $this->aiSettings = $this->getAISettings();
        $this->openaiModel = $this->aiSettings['ai_model'] ?? config('services.openai.model');
        
        // Initialize web search service
        $this->openAIWebService = app(OpenAIWebService::class);
        
        // Debug log to verify the model being used
        Log::info('NewsChatbot initialized with model: ' . $this->openaiModel);
        Log::info('Web search enabled: ' . ($this->openAIWebService->isWebSearchEnabled() ? 'Yes' : 'No'));
    }

    /**
     * Get crawler database connection
     */
    private function getCrawlerConnection()
    {
        $config = config('database.connections.crawler');
        
        return new \PDO(
            "mysql:host={$config['host']};port={$config['port']};dbname={$config['database']};charset={$config['charset']}",
            $config['username'],
            $config['password'],
            [\PDO::ATTR_ERRMODE => \PDO::ERRMODE_EXCEPTION]
        );
    }

    /**
     * Log query to database
     */
    private function logQuery($data)
    {
        try {
            QueryLog::create($data);
        } catch (\Exception $e) {
            Log::error('Failed to log query: ' . $e->getMessage());
        }
    }

    /**
     * Get default AI settings from database
     */
    private function getDefaultAISettings()
    {
        try {
            $settings = DB::table('ai_settings')
                ->where('category', 'ai')
                ->pluck('setting_value', 'setting_key')
                ->toArray();
            
            return [
                'ai_temperature' => (float) ($settings['ai_temperature'] ?? 0.2),
                'ai_max_tokens' => (int) ($settings['ai_max_tokens'] ?? 150),
                'ai_model' => $settings['ai_model'] ?? config('services.openai.model'),
                'ai_response_speed' => $settings['ai_response_speed'] ?? 'balanced'
            ];
        } catch (\Exception $e) {
            Log::error('Failed to load default AI settings from database: ' . $e->getMessage());
            // If database fails, try to get from config or use minimal defaults
            return [
                'ai_temperature' => (float) config('services.openai.temperature', 0.2),
                'ai_max_tokens' => (int) config('services.openai.max_tokens', 150),
                'ai_model' => config('services.openai.model', 'gpt-4-turbo'),
                'ai_response_speed' => config('services.openai.response_speed', 'balanced')
            ];
        }
    }

    /**
     * Get AI settings from database
     */
    private function getAISettings()
    {
        try {
            $settings = DB::table('ai_settings')
                ->where('category', 'ai')
                ->pluck('setting_value', 'setting_key')
                ->toArray();
            
            // Get defaults from database
            $defaults = $this->getDefaultAISettings();
            
            return [
                'ai_temperature' => (float) ($settings['ai_temperature'] ?? $defaults['ai_temperature']),
                'ai_max_tokens' => (int) ($settings['ai_max_tokens'] ?? $defaults['ai_max_tokens']),
                'ai_model' => $settings['ai_model'] ?? $defaults['ai_model'],
                'ai_response_speed' => $settings['ai_response_speed'] ?? $defaults['ai_response_speed']
            ];
        } catch (\Exception $e) {
            Log::error('Failed to load AI settings from database: ' . $e->getMessage());
            return $this->getDefaultAISettings();
        }
    }

    /**
     * Get UI settings from database
     */
    private function getUISettings()
    {
        try {
            $settings = DB::table('ai_settings')
                ->where('category', 'ui')
                ->pluck('setting_value', 'setting_key')
                ->toArray();
            
            return [
                'typing_speed' => $settings['typing_speed'] ?? 'medium',
                'ad_display_delay' => $settings['ad_display_delay'] ?? 'short',
                'ad_frequency' => $settings['ad_frequency'] ?? 'every',
                'auto_scroll' => $settings['auto_scroll'] ?? 'enabled',
                'sound_notifications' => $settings['sound_notifications'] ?? 'disabled'
            ];
        } catch (\Exception $e) {
            Log::error('Failed to load UI settings from database: ' . $e->getMessage());
            return [
                'typing_speed' => 'medium',
                'ad_display_delay' => 'short',
                'ad_frequency' => 'every',
                'auto_scroll' => 'enabled',
                'sound_notifications' => 'disabled'
            ];
        }
    }

    public function index()
    {
        try {
            // Get real sub-categories from the crawler database
            $pdo = $this->getCrawlerConnection();
            
            $stmt = $pdo->prepare("SELECT DISTINCT sub_category FROM vw_crawler_mews WHERE sub_category IS NOT NULL ORDER BY sub_category");
            $stmt->execute();
            $allCategories = $stmt->fetchAll(\PDO::FETCH_COLUMN);
            
            // Get UI settings for typing speed and other UI preferences
            $uiSettings = $this->getUISettings();
            
            // Define the available news interest options (matching subcategories)
            $availableNewsInterests = ['Politics', 'Business', 'Entertainment', 'Sports', 'International', 'General', 'Local', 'Technology'];
            
            // Personalize categories based on user interests
            $userInterests = [];
            $personalizedCategories = [];
            
            if (auth()->check()) {
                $user = auth()->user();
                if ($user->profile && $user->profile->interests) {
                    $userInterests = $user->profile->interests;
                    
                    // Use user's selected interests directly (they should match available options)
                    $personalizedCategories = array_intersect($userInterests, $availableNewsInterests);
                    
                    // If user has no valid interests, use default categories
                    if (empty($personalizedCategories)) {
                        $personalizedCategories = ['Politics', 'Business', 'Entertainment', 'Sports', 'International'];
                    }
                } else {
                    // User logged in but no interests set, use default categories
                    $personalizedCategories = ['Politics', 'Business', 'Entertainment', 'Sports', 'International'];
                }
            } else {
                // Not logged in, use default categories
                $personalizedCategories = ['Politics', 'Business', 'Entertainment', 'Sports', 'International'];
            }
            
            // Add debug info for development
            if (config('app.debug')) {
                Log::info('Chatbot personalization', [
                    'user_logged_in' => auth()->check(),
                    'user_interests' => $userInterests,
                    'personalized_categories' => $personalizedCategories,
                    'all_categories_count' => count($allCategories ?? [])
                ]);
            }
            
            return view('news-chatbot', compact('personalizedCategories', 'userInterests', 'availableNewsInterests', 'uiSettings'));
            
        } catch (\Exception $e) {
            // Fallback to default categories if database fails
            $personalizedCategories = ['Politics', 'Business', 'Entertainment', 'Sports', 'International'];
            $userInterests = [];
            $availableNewsInterests = ['Politics', 'Business', 'Entertainment', 'Sports', 'International', 'General', 'Local', 'Technology'];
            $uiSettings = $this->getUISettings();
            return view('news-chatbot', compact('personalizedCategories', 'userInterests', 'availableNewsInterests', 'uiSettings'));
        }
    }

    
    /**
     * Truncate content to specified length
     */
    private function truncateContent($content, $length = 200)
    {
        if (strlen($content) <= $length) {
            return $content;
        }
        
        return substr($content, 0, $length) . '...';
    }
    
    /**
     * Format date for display
     */
    private function formatDate($date)
    {
        try {
            $dateObj = new \DateTime($date);
            return $dateObj->format('M j, Y');
        } catch (\Exception $e) {
            return $date;
        }
    }

    public function chat(Request $request)
    {
        $startTime = microtime(true);
        $userQuery = $request->input('query');
        $sessionId = $request->input('session_id', 'default');
        $userId = auth()->id(); // Get authenticated user ID if available
        
        try {
            if (empty($userQuery)) {
                return response()->json([
                    'success' => false,
                    'error' => 'No query provided'
                ]);
            }

            // Debug log to verify the model being used
            Log::info('Processing chat request with model: ' . $this->openaiModel);
            
            // Step 1: Use OpenAI to intelligently analyze the query and determine intent
            $aiAnalysis = $this->analyzeQueryWithAI($userQuery, $sessionId);
            $aiAnalysis['original_query'] = $userQuery;
            $aiAnalysis['session_id'] = $sessionId;
            
            // Debug log to see what intent type was detected
            Log::info('Query: ' . $userQuery . ' - Detected intent: ' . ($aiAnalysis['intent_type'] ?? 'unknown'));

            // Step 2: Handle based on AI analysis
            if ($aiAnalysis['intent_type'] === 'election_query') {
                // Redirect to Election Intelligence Dashboard
                $response = "Great question! For in-depth election analysis, predictions, historical data, and poll tracking, I recommend our dedicated **Election Intelligence** dashboard. It has:\n\n" .
                    "🗳️ **Historical Election Data** - Results from 2000-2024 by region\n" .
                    "📊 **Poll Tracker** - Latest polls from credible sources\n" .
                    "🤖 **AI Predictions** - Win probabilities based on data\n" .
                    "🗺️ **Regional Analysis** - Voting patterns by constituency\n" .
                    "📈 **Year Comparisons** - See how voting changed over time\n\n" .
                    "Would you like me to take you there? [Go to Election Intelligence](/elections)";
                
                $newsResults = [];
                $isNewsQuery = false;
                
                // Log this as an election redirect
                Log::info('Election query detected, offering redirect: ' . $userQuery);
            } elseif ($aiAnalysis['intent_type'] === 'more_request') {
                // Handle "more" request using context
                $response = $this->handleMoreRequest($sessionId, $userQuery);
                $newsResults = $this->getAdditionalNewsForMore($sessionId);
                $isNewsQuery = true;
            } elseif ($aiAnalysis['intent_type'] === 'news_query') {
                // Handle news-related query - DB first approach
                $newsResults = $this->fetchRelevantNews($aiAnalysis);
                
                if (!empty($newsResults)) {
                    // Use database news if available
                    $response = $this->generateNewsResponse($newsResults, $aiAnalysis);
                } else {
                    // Fallback to web search if no database results
                    $response = $this->handleWebSearchQuery($userQuery);
                    $newsResults = []; // Web search doesn't return structured news results
                }
                
                // Store context for future "more" requests
                $this->storeConversationContext($sessionId, $aiAnalysis, $newsResults);
                $isNewsQuery = true;
            } else {
                // Handle general conversation - use web search for current information
                // Include page context when user is viewing a specific page (FAB chatbot)
                $pageContext = $request->input('page_context', []);
                $response = $this->handleGeneralConversation($userQuery, $pageContext);
                $newsResults = [];
                $isNewsQuery = false;
            }

            // Calculate response time
            $responseTime = round((microtime(true) - $startTime) * 1000); // Convert to milliseconds

            // Log the query to database
            $this->logQuery([
                'user_id' => $userId,
                'session_id' => $sessionId,
                'query' => $userQuery,
                'intent' => $aiAnalysis['intent_type'] ?? 'unknown',
                'response_data' => [
                    'response' => $response,
                    'is_news_query' => $isNewsQuery,
                    'news_count' => count($newsResults),
                    'analysis' => $aiAnalysis
                ],
                'response_time_ms' => $responseTime,
                'model_used' => $this->openaiModel,
                'was_successful' => true,
                'user_ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'referrer' => $request->header('referer'),
                'context_data' => $aiAnalysis
            ]);

            // Database already returns the correct structure, no transformation needed
            $transformedNewsResults = $newsResults;

            return response()->json([
                'success' => true,
                'response' => $response,
                'news_data' => $transformedNewsResults,
                'query_analysis' => $aiAnalysis,
                'is_news_query' => $isNewsQuery
            ]);

        } catch (\Exception $e) {
            $responseTime = round((microtime(true) - $startTime) * 1000);
            
            // Log failed query
            $this->logQuery([
                'user_id' => $userId,
                'session_id' => $sessionId,
                'query' => $userQuery,
                'intent' => 'error',
                'response_data' => null,
                'response_time_ms' => $responseTime,
                'model_used' => $this->openaiModel,
                'was_successful' => false,
                'error_message' => $e->getMessage(),
                'user_ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'referrer' => $request->header('referer'),
                'context_data' => null
            ]);
            
            Log::error('News chatbot error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => 'An error occurred while processing your request'
            ]);
        }
    }

    private function analyzeQueryWithAI($userQuery, $sessionId = null)
    {
        try {
            // Get conversation context to understand follow-up questions
            $context = $sessionId ? $this->getConversationContext($sessionId) : null;
            $contextInfo = '';
            
            if ($context) {
                $contextInfo = "\n\nCONVERSATION CONTEXT: The user previously asked about: " . ($context['intent'] ?? 'unknown topic') . 
                              ". Keywords: " . ($context['keywords'] ?? 'none') . 
                              ". Location: " . ($context['location'] ?? 'none');
            }
            
            // Use a regular model for AI analysis (not search model)
            $analysisModel = 'gpt-4o-mini'; // Use regular model for structured analysis
            
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->openaiApiKey,
                'Content-Type' => 'application/json',
            ])->post('https://api.openai.com/v1/chat/completions', [
                'model' => $analysisModel,
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are an intelligent query analyzer for a Ghana-focused news chatbot with access to current information. Your job is to understand user intent and classify queries accurately.

                        GHANA DEFAULT CONTEXT:
                        - This is a Ghana-focused news platform, so when users ask general questions without specifying a country, assume they mean Ghana
                        - Examples: "Who won the last election?" → "Who won the last Ghana election?", "What\'s the inflation rate?" → "What\'s Ghana\'s inflation rate?"
                        - Only use this default when NO country is explicitly mentioned
                        - If a user mentions another country (Nigeria, USA, etc.), respect that context

                        IMPORTANT: 
                        - All political leadership questions (Ghana, Nigeria, USA, etc.) should be "general_conversation" - let ChatGPT handle them
                        - Follow-up questions like "How about Ghana?" after asking about Nigeria should be "general_conversation" (not news_query)
                        - When analyzing queries about current events, politics, or recent developments, prioritize them as "news_query" so the system can search the latest database for accurate, up-to-date information.
                        - Keep your responses concise and focused.

                        Analyze the user query and determine:
                        1. **Intent Type**: 
                           - "news_query" - User wants news about a specific topic or category (even if they previously asked about something else)
                           - "election_query" - User wants election predictions, poll data, historical election results, voting patterns, or political analysis (e.g., "who will win the election", "what are the polls showing", "how did ashanti vote in 2020", "election prediction", "compare 2016 and 2020 elections")
                           - "more_request" - User wants additional/more news on the SAME topic (only use this for "more", "more news", "show me more", etc.)
                           - "general_conversation" - User wants to chat, ask questions, or get help (including all political leadership questions for any country)
                        
                        ELECTION QUERY DETECTION:
                        - Any query about election predictions, who will win, polling data, historical election analysis, voting patterns, or swing constituencies should be "election_query"
                        - Examples: "Who will win 2024?", "What do polls say?", "How did Volta vote?", "NPP vs NDC", "election forecast", "predict the election"
                        
                        IMPORTANT: Only classify as more_request if the user explicitly asks for more without specifying a new topic. If they ask for news on a specific category (even if they previously asked about something else), it is a news_query.
                        
                        2. **Category**: The main news category (sports, politics, technology, business, entertainment, health, science, general, international, local)
                        3. **Keywords**: Specific search terms that will find the most relevant articles (include location names as keywords if mentioned)
                        4. **Intent**: What type of news the user wants (breaking news, analysis, updates, trends, etc.)
                        5. **Timeframe**: How recent the news should be (latest, today, this week, recent)
                        
                        Respond with JSON format: {
                            \"intent_type\": \"news_query|more_request|general_conversation\",
                            \"category\": \"category_name\",
                            \"keywords\": \"search_keywords\", 
                            \"intent\": \"user_intent_description\",
                            \"timeframe\": \"recent/latest/today/week/month\"
                        }
                        
                        Examples:
                        - \"What\'s new in sports?\" → {\"intent_type\": \"news_query\", \"category\": \"sports\", \"keywords\": \"ghana sports news latest\", \"intent\": \"latest Ghana sports updates and scores\", \"timeframe\": \"recent\"}
                        - \"What\'s the latest news in Politics?\" → {\"intent_type\": \"news_query\", \"category\": \"politics\", \"keywords\": \"ghana politics news latest\", \"intent\": \"latest Ghana political news and updates\", \"timeframe\": \"recent\"}
                        - \"What\'s the latest news in Business?\" → {\"intent_type\": \"news_query\", \"category\": \"business\", \"keywords\": \"ghana business news latest\", \"intent\": \"latest Ghana business news and updates\", \"timeframe\": \"recent\"}
                        - \"What\'s the latest news in Entertainment?\" → {\"intent_type\": \"news_query\", \"category\": \"entertainment\", \"keywords\": \"ghana entertainment news latest\", \"intent\": \"latest Ghana entertainment news and updates\", \"timeframe\": \"recent\"}
                        - \"What\'s the latest news in general?\" → {\"intent_type\": \"news_query\", \"category\": \"general\", \"keywords\": \"ghana general news latest\", \"intent\": \"latest Ghana general news updates\", \"timeframe\": \"recent\"}
                        - \"Who won the last election?\" → {\"intent_type\": \"election_query\", \"category\": \"politics\", \"keywords\": \"ghana election results 2024\", \"intent\": \"Ghana election results information\", \"timeframe\": \"recent\"}
                        - \"Who will win the 2024 election?\" → {\"intent_type\": \"election_query\", \"category\": \"politics\", \"keywords\": \"ghana election prediction 2024\", \"intent\": \"election prediction request\", \"timeframe\": null}
                        - \"What do the polls say?\" → {\"intent_type\": \"election_query\", \"category\": \"politics\", \"keywords\": \"ghana election polls\", \"intent\": \"polling data request\", \"timeframe\": \"recent\"}
                        - \"How did Ashanti vote in 2020?\" → {\"intent_type\": \"election_query\", \"category\": \"politics\", \"keywords\": \"ashanti region 2020 election results\", \"intent\": \"historical election data\", \"timeframe\": null}
                        - \"Compare NPP and NDC\" → {\"intent_type\": \"election_query\", \"category\": \"politics\", \"keywords\": \"npp ndc comparison\", \"intent\": \"party comparison request\", \"timeframe\": null}
                        - \"What\'s the inflation rate?\" → {\"intent_type\": \"general_conversation\", \"category\": null, \"keywords\": \"ghana inflation rate current\", \"intent\": \"Ghana economic information\", \"timeframe\": \"recent\"}
                        - \"Who is the president?\" → {\"intent_type\": \"general_conversation\", \"category\": null, \"keywords\": \"ghana president current\", \"intent\": \"Ghana political leadership information\", \"timeframe\": \"recent\"}
                        - \"Who is the president of Ghana?\" → {\"intent_type\": \"general_conversation\", \"category\": null, \"keywords\": \"ghana president current\", \"intent\": \"current political leadership information\", \"timeframe\": \"recent\"}
                        - \"Who is the president of Nigeria?\" → {\"intent_type\": \"general_conversation\", \"category\": null, \"keywords\": \"nigeria president current\", \"intent\": \"current political leadership information\", \"timeframe\": \"recent\"}
                        - \"Who rules Nigeria?\" → {\"intent_type\": \"general_conversation\", \"category\": null, \"keywords\": \"nigeria leadership current\", \"intent\": \"current political leadership information\", \"timeframe\": null}
                        - \"more\" → {\"intent_type\": \"more_request\", \"category\": null, \"keywords\": null, \"intent\": \"user wants more news on previous topic\", \"timeframe\": null}
                        - \"give me news on sports\" → {\"intent_type\": \"news_query\", \"category\": \"sports\", \"keywords\": \"ghana sports news latest\", \"intent\": \"latest Ghana sports news and updates\", \"timeframe\": \"recent\"}
                        - \"imme news on sports\" → {\"intent_type\": \"news_query\", \"category\": \"sports\", \"keywords\": \"ghana sports news latest\", \"intent\": \"latest Ghana sports news and updates\", \"timeframe\": \"recent\"}
                        - \"How are you?\" → {\"intent_type\": \"general_conversation\", \"category\": null, \"keywords\": null, \"intent\": \"general greeting\", \"timeframe\": null}
                        - \"What\'s happening in Kumasi?\" → {\"intent_type\": \"news_query\", \"category\": \"local\", \"keywords\": \"kumasi ghana\", \"intent\": \"local news from Kumasi, Ghana\", \"timeframe\": \"recent\"}
                        - \"gimme news on Hearts of Oak\" → {\"intent_type\": \"news_query\", \"category\": \"sports\", \"keywords\": \"hearts of oak ghana\", \"intent\": \"news about Hearts of Oak football club in Ghana\", \"timeframe\": \"recent\"}'
                    ],
                    [
                        'role' => 'user',
                        'content' => $userQuery . $contextInfo
                    ]
                ],
                'max_tokens' => $this->aiSettings['ai_max_tokens'],
                'temperature' => $this->aiSettings['ai_temperature']
            ]);

            if ($response->successful()) {
                $data = $response->json();
                $aiResponse = $data['choices'][0]['message']['content'] ?? '';
                
                // Debug log to see what AI is returning
                Log::info('AI Response for query: ' . $userQuery . ' - Response: ' . $aiResponse);
                
                // Try to parse JSON response
                $parsed = $this->parseAIResponse($aiResponse);
                if ($parsed) {
                    Log::info('Parsed AI response: ' . json_encode($parsed));
                    return $parsed;
                }
            }

            // Fallback: simple keyword extraction
            return $this->extractKeywordsFallback($userQuery);

        } catch (\Exception $e) {
            Log::error('OpenAI API error: ' . $e->getMessage());
            return $this->extractKeywordsFallback($userQuery);
        }
    }

    private function parseAIResponse($aiResponse)
    {
        $jsonStart = strpos($aiResponse, '{');
        $jsonEnd = strrpos($aiResponse, '}');
        
        if ($jsonStart !== false && $jsonEnd !== false) {
            $jsonStr = substr($aiResponse, $jsonStart, $jsonEnd - $jsonStart + 1);
            $parsed = json_decode($jsonStr, true);
            
            if ($parsed && isset($parsed['intent_type'])) {
                return $parsed;
            }
        }
        
        return null;
    }

    private function extractKeywordsFallback($query)
    {
        $queryLower = strtolower(trim($query));
        
        // Check if it's a "more" request
        if (in_array($queryLower, ['more', 'more news', 'show more', 'additional', 'extra', 'continue', 'what else'])) {
            return [
                'intent_type' => 'more_request',
                'category' => null,
                'keywords' => null,
                'intent' => 'user wants more news on previous topic',
                'timeframe' => null,
                'location' => null
            ];
        }
        
        // Check if it's a news query
        $newsKeywords = ['news', 'latest', 'what\'s happening', 'what\'s new', 'updates', 'breaking', 'sports', 'politics', 'technology', 'business', 'entertainment', 'health', 'science'];
        
        // Check for follow-up questions that should be general conversation
        if (in_array($queryLower, ['how about', 'what about', 'and', 'also', 'too', 'as well'])) {
            return [
                'intent_type' => 'general_conversation',
                'category' => null,
                'keywords' => 'follow-up question',
                'intent' => 'follow-up question about previous topic',
                'timeframe' => null
            ];
        }
        
        foreach ($newsKeywords as $keyword) {
            if (strpos($queryLower, $keyword) !== false) {
                return [
                    'intent_type' => 'news_query',
                    'category' => $keyword,
                    'keywords' => $keyword,
                    'intent' => 'news search',
                    'timeframe' => 'recent'
                ];
            }
        }
        
        // Check for location names as keywords (treat as news query)
        $locationKeywords = ['akwatia', 'ghana', 'accra', 'kumasi', 'tamale', 'cape coast', 'sunyani', 'ho', 'wa', 'bolgatanga'];
        foreach ($locationKeywords as $location) {
            if (strpos($queryLower, $location) !== false) {
                return [
                    'intent_type' => 'news_query',
                    'category' => 'local',
                    'keywords' => $location,
                    'intent' => 'local news from ' . $location,
                    'timeframe' => 'recent'
                ];
            }
        }
        
        // Default to general conversation
        return [
            'intent_type' => 'general_conversation',
            'category' => null,
            'keywords' => null,
            'intent' => 'general conversation',
            'timeframe' => null
        ];
    }

    private function handleMoreRequest($sessionId, $userQuery)
    {
        $context = $this->getConversationContext($sessionId);
        
        if (!$context) {
            return "I don't have any previous context to provide more news about. Please ask me a specific question about what you'd like to know more about!";
        }
        
        // Generate response for additional news
        $response = $this->generateAdditionalNewsResponse([], $context); // Empty array since we'll get news separately
        
        return $response;
    }

    private function getAdditionalNewsForMore($sessionId)
    {
        $context = $this->getConversationContext($sessionId);
        
        if (!$context) {
            return [];
        }
        
        // Fetch additional news based on the previous context
        $additionalNews = $this->fetchAdditionalNews($context);
        
        if (!empty($additionalNews)) {
            // Update context with new results
            $this->updateConversationContext($sessionId, $additionalNews);
        }
        
        return $additionalNews;
    }

    private function fetchAdditionalNews($context)
    {
        try {
            $pdo = $this->getCrawlerConnection();
            
            $excludeIds = $context['shown_article_ids'] ?? [];
            $excludeIdsStr = !empty($excludeIds) ? implode(',', $excludeIds) : '0';
            
            // Build query based on previous context
            if (!empty($context['sub_category'])) {
                $stmt = $pdo->prepare("
                    SELECT * FROM vw_crawler_mews 
                    WHERE LOWER(sub_category) LIKE LOWER(:sub_category)
                    AND post_id NOT IN ({$excludeIdsStr})
                    ORDER BY date_added DESC 
                    LIMIT 3
                ");
                $stmt->execute(['sub_category' => '%' . $context['sub_category'] . '%']);
            } else {
                // Fallback to keyword search (title only)
                $stmt = $pdo->prepare("
                    SELECT * FROM vw_crawler_mews 
                    WHERE LOWER(post_title) LIKE LOWER(:keywords)
                    AND post_id NOT IN ({$excludeIdsStr})
                    ORDER BY date_added DESC 
                    LIMIT 3
                ");
                $stmt->execute(['keywords' => '%' . ($context['keywords'] ?? 'news') . '%']);
            }
            
            $newsResults = $stmt->fetchAll(\PDO::FETCH_ASSOC);
            
            // Format the results
            $formattedResults = [];
            foreach ($newsResults as $article) {
                $formattedResults[] = [
                    'id' => $article['post_id'] ?? 0,
                    'title' => $article['post_title'] ?? 'Untitled Article',
                    'category' => $article['category'] ?? 'General',
                    'sub_category' => $article['sub_category'] ?? '',
                    'content' => $this->cleanContent($article['post_content'] ?? '', 150),
                    'date' => $this->formatDate($article['date_added'] ?? time()),
                    'source' => $article['post_source'] ?? '#',
                    'image' => $article['image'] ?? '',
                    'views' => $article['views'] ?? 0
                ];
            }
            
            return $formattedResults;
            
        } catch (\Exception $e) {
            Log::error('Database error in fetchAdditionalNews: ' . $e->getMessage());
            return [];
        }
    }

    private function generateAdditionalNewsResponse($additionalNews, $context)
    {
        try {
            $articlesSummary = '';
            foreach ($additionalNews as $article) {
                $articlesSummary .= "Title: {$article['title']}\n";
                $articlesSummary .= "Content: {$article['content']}\n";
                $articlesSummary .= "Date: {$article['date']}\n\n";
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->openaiApiKey,
                'Content-Type' => 'application/json',
            ])->post('https://api.openai.com/v1/chat/completions', [
                'model' => $this->openaiModel,
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a professional news anchor. The user asked for more news on the same topic. Present these additional articles in a natural, engaging way. Use phrases like "Here\'s more on that," "In additional news," "Also worth noting," etc. Keep it concise and professional.'
                    ],
                    [
                        'role' => 'user',
                        'content' => "User asked for more news about: {$context['intent']}. Here are additional articles:\n\n{$articlesSummary}\n\nPlease provide a professional news caster response for these additional articles."
                    ]
                ],
                'max_tokens' => $this->aiSettings['ai_max_tokens'],
                'temperature' => $this->aiSettings['ai_temperature']
            ]);

            if ($response->successful()) {
                $data = $response->json();
                return $data['choices'][0]['message']['content'] ?? '';
            }

        } catch (\Exception $e) {
            Log::error('OpenAI response generation error: ' . $e->getMessage());
        }

        // Fallback response
        $response = "Here's more news for you:\n\n";
        foreach ($additionalNews as $article) {
            $response .= "📰 **{$article['title']}** - {$article['content']}\n\n";
        }
        
        return $response;
    }

    /**
     * Handle queries with web search capability (fallback only)
     */
    private function handleWebSearchQuery($userQuery)
    {
        try {
            // Check if web search is enabled in settings
            if (!$this->isWebSearchEnabled()) {
                // Web search is disabled, use regular conversation
                return $this->handleGeneralConversation($userQuery);
            }
            
            // Use the web search service for real-time information
            $response = $this->openAIWebService->generateNewsResponse($userQuery);
            
            if ($response && !empty($response['content'])) {
                // Source references are already removed by the service
                return $response['content'];
            }
            
            // Fallback to regular conversation if web search fails
            return $this->handleGeneralConversation($userQuery);
            
        } catch (\Exception $e) {
            Log::error('Web search query error: ' . $e->getMessage());
            return $this->handleGeneralConversation($userQuery);
        }
    }

    /**
     * Load election/census/study context for the chatbot (elections page or census/population questions).
     */
    private function getElectionCensusContext(): string
    {
        $path = 'chatbot_context/election_census_study.txt';
        $base = Storage::disk('local')->exists($path)
            ? trim(Storage::disk('local')->get($path))
            : '';

        $lines = [];
        if ($base !== '') {
            $lines[] = $base;
        }

        $region2021 = RegionCensus::where('census_year', 2021)->with('region')->get();
        if ($region2021->isNotEmpty()) {
            $total = $region2021->sum('population');
            $lines[] = "\n2021 PHC regional population (Ghana): Total " . number_format($total) . ". Top regions by population: " .
                $region2021->sortByDesc('population')->take(8)->map(function ($rc) {
                    return $rc->region?->name . ' ' . number_format($rc->population);
                })->join('; ');
        }

        $text = implode("\n", $lines);
        return $text !== '' ? "\n\nELECTION & CENSUS CONTEXT (use for population/election/demographic questions):\n" . substr($text, 0, 3500) : '';
    }

    /**
     * Whether the user is on an election-related page or asking about census/population/demographics.
     */
    private function needsElectionCensusContext(array $pageContext, string $userQuery): bool
    {
        $url = $pageContext['url'] ?? '';
        if (str_contains(strtolower($url), 'election')) {
            return true;
        }
        $q = strtolower($userQuery);
        $keywords = ['census', 'population', 'demographic', 'voting population', 'regional population', 'phc', 'how many people', 'voter population'];
        foreach ($keywords as $kw) {
            if (str_contains($q, $kw)) {
                return true;
            }
        }
        return false;
    }

    private function handleGeneralConversation($userQuery, array $pageContext = [])
    {
        try {
            // First, check if the query is off-topic and needs redirection
            $isOffTopic = $this->isOffTopicQuery($userQuery);
            
            if ($isOffTopic) {
                return $this->generateOffTopicResponse($userQuery);
            }
            
            // Check if this is a question that needs current information
            $needsCurrentInfo = $this->needsCurrentInformation($userQuery);
            
            if ($needsCurrentInfo && $this->isWebSearchEnabled() && $this->openAIWebService->isWebSearchEnabled()) {
                // Use web search for current information
                $response = $this->openAIWebService->generateNewsResponse($userQuery);
                if ($response && !empty($response['content'])) {
                    return $response['content'];
                }
            }
            
            // Build system prompt with optional page context
            $systemContent = 'You are a professional Ghana-focused news assistant and journalist. Your primary role is to help users with news-related queries, current events, and general knowledge questions.

GHANA DEFAULT CONTEXT:
- This is a Ghana-focused news platform, so when users ask general questions without specifying a country, assume they mean Ghana
- Examples: "Who won the last election?" → "Who won the last Ghana election?", "What\'s the inflation rate?" → "What\'s Ghana\'s inflation rate?"
- Only use this default when NO country is explicitly mentioned
- If a user mentions another country (Nigeria, USA, etc.), respect that context

IMPORTANT GUIDELINES:
- You are a NEWS ASSISTANT who can also answer general knowledge questions.
- For current events, politics, business, sports, technology, and other news topics, provide detailed, up-to-date information.
- For general knowledge questions (like "Who is the president?"), provide accurate, current information about Ghana by default.
- If users ask about non-news topics (food, personal advice, entertainment recommendations, etc.), politely redirect them back to news topics.
- Always stay factual and never hallucinate or make up information.
- If you don\'t know something, say so clearly.
- Keep responses concise and professional.
- Focus on providing helpful, accurate information.

When users go off-topic, gently guide them back to news with phrases like:
- "I\'m a news assistant, so I can help you with current events instead..."
- "Let me help you with the latest news on..."
- "I focus on news and current events. Would you like to know about..."

Be helpful but stay within your role as a news assistant.';

            // Add page context when user is viewing a specific page (FAB chatbot)
            if (!empty($pageContext['url']) || !empty($pageContext['content'])) {
                $systemContent .= "\n\nPAGE CONTEXT: The user is currently viewing a page.";
                if (!empty($pageContext['url'])) {
                    $systemContent .= "\n- Page URL: " . $pageContext['url'];
                }
                if (!empty($pageContext['title'])) {
                    $systemContent .= "\n- Page Title: " . $pageContext['title'];
                }
                if (!empty($pageContext['content'])) {
                    $content = is_string($pageContext['content']) ? $pageContext['content'] : json_encode($pageContext['content']);
                    $content = substr($content, 0, 4000); // Limit to avoid token overflow
                    $systemContent .= "\n- Page Content (summary): " . $content;
                }
                $systemContent .= "\n\nUse this context to explain, decipher, or summarize what the user is viewing when they ask questions like 'explain this', 'what does this mean', 'help me understand', or 'summarize this page'.";
            }

            if ($this->needsElectionCensusContext($pageContext, $userQuery)) {
                $systemContent .= $this->getElectionCensusContext();
            }
            
            // Use a regular model for general conversation (not search model)
            $conversationModel = 'gpt-4o-mini';
            
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->openaiApiKey,
                'Content-Type' => 'application/json',
            ])->post('https://api.openai.com/v1/chat/completions', [
                'model' => $conversationModel,
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => $systemContent
                    ],
                    [
                        'role' => 'user',
                        'content' => $userQuery
                    ]
                ],
                'max_tokens' => $this->aiSettings['ai_max_tokens'],
                'temperature' => 0.3 // Lower temperature for more factual responses
            ]);

            if ($response->successful()) {
                $data = $response->json();
                return $data['choices'][0]['message']['content'] ?? '';
            }

        } catch (\Exception $e) {
            Log::error('OpenAI general conversation error: ' . $e->getMessage());
        }

        return "I'm here to help! I can answer general questions and help you find news. What would you like to know?";
    }

    private function isOffTopicQuery($userQuery)
    {
        $offTopicKeywords = [
            // Food and dining
            'hungry', 'food', 'eat', 'restaurant', 'cooking', 'recipe', 'meal', 'lunch', 'dinner', 'breakfast',
            'pizza', 'burger', 'chicken', 'rice', 'soup', 'salad', 'pasta', 'sandwich',
            
            // Personal advice and relationships
            'relationship', 'dating', 'boyfriend', 'girlfriend', 'marriage', 'divorce', 'love', 'romance',
            'advice', 'personal', 'therapy', 'counseling', 'depression', 'anxiety', 'stress',
            
            // Entertainment recommendations
            'movie', 'film', 'show', 'series', 'netflix', 'youtube', 'music', 'song', 'artist', 'album',
            'game', 'gaming', 'play', 'fun', 'entertainment', 'hobby', 'leisure',
            
            // Shopping and products
            'buy', 'purchase', 'shop', 'shopping', 'product', 'price', 'cost', 'deal', 'sale',
            'clothes', 'fashion', 'shoes', 'phone', 'laptop', 'car', 'house',
            
            // Health and medical (non-news)
            'sick', 'illness', 'medicine', 'drug', 'pain', 'headache', 'fever', 'cough',
            'doctor', 'hospital', 'medical', 'treatment', 'symptoms',
            
            // Travel and vacation
            'vacation', 'holiday', 'travel', 'trip', 'hotel', 'flight', 'booking', 'reservation',
            'beach', 'mountains', 'city', 'country', 'passport', 'visa',
            
            // General off-topic
            'weather', 'joke', 'funny', 'meme', 'gossip', 'celebrity', 'rumor', 'story',
            'dream', 'nightmare', 'sleep', 'tired', 'bored', 'lonely'
        ];
        
        $queryLower = strtolower($userQuery);
        
        // Check for off-topic keywords
        foreach ($offTopicKeywords as $keyword) {
            if (strpos($queryLower, $keyword) !== false) {
                return true;
            }
        }
        
        // Check for common off-topic patterns
        $offTopicPatterns = [
            '/what.*should.*i.*eat/i',
            '/where.*should.*i.*go/i',
            '/what.*movie.*should.*i.*watch/i',
            '/recommend.*me/i',
            '/i.*am.*hungry/i',
            '/i.*want.*to.*eat/i',
            '/what.*do.*you.*think.*about/i',
            '/give.*me.*advice/i',
            '/help.*me.*with.*my/i',
            '/i.*feel.*(sad|lonely|bored|tired)/i'
        ];
        
        foreach ($offTopicPatterns as $pattern) {
            if (preg_match($pattern, $queryLower)) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Check if a query needs current information
     */
    private function needsCurrentInformation($userQuery)
    {
        $currentInfoKeywords = [
            // Political leadership
            'president', 'vice president', 'prime minister', 'minister', 'governor', 'mayor',
            'who is', 'who are', 'current', 'now', 'today', 'latest',
            
            // Current events
            'happening', 'going on', 'recent', 'latest news', 'breaking',
            
            // Time-sensitive information
            '2024', '2025', 'this year', 'this month', 'this week',
            
            // Financial/Currency information
            'exchange rate', 'currency', 'usd', 'ghs', 'dollar', 'cedi', 'rate', 'price',
            'inflation', 'interest rate', 'stock', 'market', 'trading',
            
            // Country-specific current info
            'ghana', 'nigeria', 'usa', 'united states', 'uk', 'united kingdom'
        ];
        
        $queryLower = strtolower($userQuery);
        
        foreach ($currentInfoKeywords as $keyword) {
            if (strpos($queryLower, $keyword) !== false) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Check if web search is enabled in settings
     */
    private function isWebSearchEnabled()
    {
        try {
            $webSearchEnabled = DB::table('ai_settings')
                ->where('setting_key', 'ai_web_search_enabled')
                ->value('setting_value');
            
            return $webSearchEnabled === 'true';
        } catch (\Exception $e) {
            Log::error('Failed to check web search setting: ' . $e->getMessage());
            return true; // Default to enabled if setting not found
        }
    }

    private function generateOffTopicResponse($userQuery)
    {
        $responses = [
            "I'm a news assistant, so I focus on current events and news topics. Let me help you with the latest news instead! What's happening in your area or what news topics interest you?",
            
            "I specialize in news and current events. Instead of that, would you like to know about the latest developments in politics, business, sports, or technology?",
            
            "As a news assistant, I can help you stay informed about current events. What news topics would you like to explore? I can share the latest updates on various subjects.",
            
            "I'm designed to help with news and current events. Let me redirect you to something more relevant - what's the latest news you'd like to know about?",
            
            "I focus on news and journalism. How about we discuss current events instead? What news topics are you interested in following?"
        ];
        
        // Add a specific suggestion based on the query
        $queryLower = strtolower($userQuery);
        
        if (strpos($queryLower, 'hungry') !== false || strpos($queryLower, 'food') !== false || strpos($queryLower, 'eat') !== false) {
            return "I'm a news assistant, so I can't help with food recommendations. But I can tell you about food industry news, restaurant business updates, or agricultural developments! What interests you?";
        }
        
        if (strpos($queryLower, 'movie') !== false || strpos($queryLower, 'entertainment') !== false) {
            return "I focus on news rather than entertainment recommendations. But I can share the latest entertainment industry news, celebrity updates, or media business developments! What would you like to know?";
        }
        
        if (strpos($queryLower, 'advice') !== false || strpos($queryLower, 'help') !== false) {
            return "I'm a news assistant, so I can't provide personal advice. But I can help you stay informed about current events and news that might be relevant to your interests! What news topics interest you?";
        }
        
        // Return a random general response
        return $responses[array_rand($responses)];
    }

    private function fetchRelevantNews($aiAnalysis)
    {
        try {
            // Connect to the crawler database
            $pdo = $this->getCrawlerConnection();
            
            $newsResults = [];
            
            // Priority 1: Keyword search first with strict title/category matching
            if (!empty($aiAnalysis['keywords'])) {
                $stmt = $pdo->prepare("
                    SELECT *, 
                    CASE 
                        WHEN LOWER(post_title) LIKE LOWER(:keywords) THEN 4
                        WHEN LOWER(sub_category) LIKE LOWER(:keywords) THEN 3
                        WHEN LOWER(category) LIKE LOWER(:keywords) THEN 3
                        ELSE 0
                    END as relevance_score
                    FROM vw_crawler_mews 
                    WHERE LOWER(post_title) LIKE LOWER(:keywords) 
                    OR LOWER(sub_category) LIKE LOWER(:keywords)
                    OR LOWER(category) LIKE LOWER(:keywords)
                    ORDER BY relevance_score DESC, date_added DESC 
                    LIMIT 5
                ");
                $searchTerm = '%' . $aiAnalysis['keywords'] . '%';
                $stmt->execute(['keywords' => $searchTerm]);
                $newsResults = $stmt->fetchAll(\PDO::FETCH_ASSOC);
            }
            
            // Priority 2: Category-based search if no keyword results
            if (empty($newsResults) && !empty($aiAnalysis['category'])) {
                $stmt = $pdo->prepare("
                    SELECT * FROM vw_crawler_mews 
                    WHERE LOWER(sub_category) LIKE LOWER(:category) 
                    OR LOWER(category) LIKE LOWER(:category)
                    ORDER BY date_added DESC 
                    LIMIT 5
                ");
                $stmt->execute(['category' => '%' . $aiAnalysis['category'] . '%']);
                $newsResults = $stmt->fetchAll(\PDO::FETCH_ASSOC);
            }
            
            // Priority 3: Fallback to latest news if no specific results
            if (empty($newsResults)) {
                $stmt = $pdo->prepare("
                    SELECT * FROM vw_crawler_mews 
                    ORDER BY date_added DESC 
                    LIMIT 5
                ");
                $stmt->execute();
                $newsResults = $stmt->fetchAll(\PDO::FETCH_ASSOC);
            }
            
            // Format the results
            $formattedResults = [];
            foreach ($newsResults as $article) {
                $formattedResults[] = [
                    'id' => $article['post_id'] ?? 0,
                    'title' => $article['post_title'] ?? 'Untitled Article',
                    'category' => $article['category'] ?? 'General',
                    'sub_category' => $article['sub_category'] ?? '',
                    'content' => $this->cleanContent($article['post_content'] ?? '', 150),
                    'date' => $this->formatDate($article['date_added'] ?? time()),
                    'source' => $article['post_source'] ?? '#',
                    'image' => $article['image'] ?? '',
                    'views' => $article['views'] ?? 0
                ];
            }
            
            return $formattedResults;
            
        } catch (\Exception $e) {
            Log::error('Database error: ' . $e->getMessage());
            
            // Fallback to sample data if database fails
            return [
                [
                    'id' => 0,
                    'title' => 'Database Connection Issue',
                    'category' => 'System',
                    'sub_category' => '',
                    'content' => 'Unable to fetch news from database. Please check your connection.',
                    'date' => now()->format('F j, Y'),
                    'source' => '#',
                    'image' => '',
                    'views' => 0
                ]
            ];
        }
    }
    
    private function cleanContent($content, $maxLength = 200)
    {
        // Remove HTML tags
        $cleanContent = strip_tags($content);
        
        // Remove common attribution patterns that aren't part of the main content
        $cleanContent = preg_replace('/Send your news stories to.*$/i', '', $cleanContent);
        $cleanContent = preg_replace('/Read Full Story.*$/i', '', $cleanContent);
        $cleanContent = preg_replace('/Advertise Here.*$/i', '', $cleanContent);
        $cleanContent = preg_replace('/Latest News.*$/i', '', $cleanContent);
        $cleanContent = preg_replace('/contact ads\[@\].*$/i', '', $cleanContent);
        
        // Clean up extra whitespace
        $cleanContent = preg_replace('/\s+/', ' ', $cleanContent);
        $cleanContent = trim($cleanContent);
        
        // Create a proper summary by finding the first complete sentence
        if (strlen($cleanContent) > $maxLength) {
            // Find the first sentence that ends with proper punctuation
            $sentences = preg_split('/([.!?]+)/', $cleanContent, -1, PREG_SPLIT_DELIM_CAPTURE);
            $summary = '';
            
            for ($i = 0; $i < count($sentences); $i += 2) {
                if (isset($sentences[$i]) && isset($sentences[$i + 1])) {
                    $sentence = $sentences[$i] . $sentences[$i + 1];
                    if (strlen($summary . $sentence) <= $maxLength) {
                        $summary .= $sentence;
                    } else {
                        break;
                    }
                }
            }
            
            // If no complete sentence fits, truncate at word boundary
            if (empty($summary)) {
                $summary = substr($cleanContent, 0, $maxLength);
                $lastSpace = strrpos($summary, ' ');
                if ($lastSpace !== false) {
                    $summary = substr($summary, 0, $lastSpace);
                }
                $summary .= '...';
            }
            
            return $summary;
        }
        
        return $cleanContent;
    }

    private function generateNewsResponse($newsResults, $aiAnalysis)
    {
        if (empty($newsResults)) {
            return "I couldn't find any relevant news articles for your query. Please try rephrasing your question or ask about a different topic.";
        }

        try {
            // Check if this is a location-based query
            if (!empty($aiAnalysis['location'])) {
                $location = $aiAnalysis['location'];
                
                // Use ChatGPT to generate a professional news caster response with location context
                $articlesSummary = '';
                foreach ($newsResults as $article) {
                    $articlesSummary .= "Title: {$article['title']}\n";
                    $articlesSummary .= "Category: {$article['category']}\n";
                    $articlesSummary .= "Sub-Category: {$article['sub_category']}\n";
                    $articlesSummary .= "Content: {$article['content']}\n";
                    $articlesSummary .= "Date: {$article['date']}\n\n";
                }

                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $this->openaiApiKey,
                    'Content-Type' => 'application/json',
                ])->post('https://api.openai.com/v1/chat/completions', [
                    'model' => $this->openaiModel,
                    'messages' => [
                        [
                            'role' => 'system',
                            'content' => 'You are a professional news anchor and broadcaster. Your job is to deliver news in a clear, engaging, and human way - just like a real news caster would on TV or radio.

                            **CRITICAL ANTI-HALLUCINATION RULES:**
                            - ONLY use information provided in the news articles below
                            - NEVER make up facts, dates, names, or details not in the articles
                            - If information is missing, say "Information not available" rather than guessing
                            - NEVER add context or background not explicitly stated in the articles
                            - If asked about something not covered in the articles, say "I don\'t have that information in the current news articles"
                            - ONLY reference articles that are actually provided in the list below

                            **Your Style:**
                            - Sound natural and conversational, not robotic
                            - Be concise but informative (2-3 sentences max)
                            - Use active voice and present tense when possible
                            - Include ONLY key facts from the provided articles
                            - Sound confident and authoritative
                            - Use transition phrases like "Meanwhile," "In other news," "Also worth noting"
                            
                            **Format for Location-Based News:**
                            - FIRST: Answer the user\'s specific question directly (only if covered in articles)
                            - THEN: Provide 2-3 bullet points of current developments from the ACTUAL articles provided
                            - Use bullet points or short phrases, not full sentences
                            - Focus ONLY on facts from the provided articles
                            - ONLY mention articles that are actually in the list below
                            
                            **CRITICAL: You MUST only reference the exact articles provided below. Do not mention any articles not in the list.**
                            
                            Provide direct, factual information ONLY from the provided articles.'
                        ],
                        [
                            'role' => 'user',
                            'content' => "User asked about news in {$location}. Here are the relevant news articles:\n\n{$articlesSummary}\n\nCRITICAL: You MUST answer the user's question first, then provide context. If they ask about Ghana's president, start with: 'Current President: John Dramani Mahama (inaugurated January 2025).' Then add: 'Latest developments: [point 1] [point 2] [point 3]'"
                        ]
                    ],
                    'max_tokens' => $this->aiSettings['ai_max_tokens'],
                    'temperature' => $this->aiSettings['ai_temperature']
                ]);

                if ($response->successful()) {
                    $data = $response->json();
                    $aiResponse = $data['choices'][0]['message']['content'] ?? '';
                    return $aiResponse;
                }
            } else {
                // Regular topic-based news response
                $articlesSummary = '';
                foreach ($newsResults as $article) {
                    $articlesSummary .= "Title: {$article['title']}\n";
                    $articlesSummary .= "Category: {$article['category']}\n";
                    $articlesSummary .= "Sub-Category: {$article['sub_category']}\n";
                    $articlesSummary .= "Content: {$article['content']}\n";
                    $articlesSummary .= "Date: {$article['date']}\n\n";
                }

                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $this->openaiApiKey,
                    'Content-Type' => 'application/json',
                ])->post('https://api.openai.com/v1/chat/completions', [
                    'model' => $this->openaiModel,
                    'messages' => [
                        [
                            'role' => 'system',
                            'content' => 'You are a professional news anchor and broadcaster. Your job is to deliver news in a clear, engaging, and human way - just like a real news caster would on TV or radio.

                            **CRITICAL ANTI-HALLUCINATION RULES:**
                            - ONLY use information provided in the news articles below
                            - NEVER make up facts, dates, names, or details not in the articles
                            - If information is missing, say "Information not available" rather than guessing
                            - NEVER add context or background not explicitly stated in the articles
                            - If asked about something not covered in the articles, say "I don\'t have that information in the current news articles"
                            - ONLY reference articles that are actually provided in the list below

                            **Your Style:**
                            - Sound natural and conversational, not robotic
                            - Be concise but informative (2-3 sentences max)
                            - Use active voice and present tense when possible
                            - Include ONLY key facts from the provided articles
                            - Sound confident and authoritative
                            - Use transition phrases like "Meanwhile," "In other news," "Also worth noting"
                            
                            **Format:**
                            - Answer the question directly and concisely (only if covered in articles)
                            - Provide a brief summary of key developments from the ACTUAL articles provided
                            - Use bullet points or short phrases, not full sentences
                            - Focus ONLY on facts from the provided articles
                            - ONLY mention articles that are actually in the list below
                            
                            **Example Style:**
                            "Latest in [category]: • [key point 1] • [key point 2] • [key point 3]"
                            
                            **IMPORTANT: Be extremely concise. Use bullet points or short phrases. Avoid full sentences and storytelling. Maximum 2-3 bullet points. ONLY reference the exact articles provided below.**
                            
                            Provide direct, factual information ONLY from the provided articles.'
                        ],
                        [
                            'role' => 'user',
                            'content' => "User asked: '{$aiAnalysis['intent']}'. Here are the relevant news articles:\n\n{$articlesSummary}\n\nCRITICAL: You MUST answer the user's question first, then provide context. If they ask about Ghana's president, start with: 'Current President: John Dramani Mahama (inaugurated January 2025).' Then add: 'Latest developments: [point 1] [point 2] [point 3]'"
                        ]
                    ],
                    'max_tokens' => $this->aiSettings['ai_max_tokens'],
                    'temperature' => $this->aiSettings['ai_temperature']
                ]);

                if ($response->successful()) {
                    $data = $response->json();
                    $aiResponse = $data['choices'][0]['message']['content'] ?? '';
                    return $aiResponse;
                }
            }

        } catch (\Exception $e) {
            Log::error('OpenAI response generation error: ' . $e->getMessage());
        }

        // Fallback to professional manual response if ChatGPT fails
        $response = $this->generateProfessionalFallback($newsResults, $aiAnalysis);
        
        return $response;
    }

    private function generateLocationSummary($location)
    {
        $summaries = [
            'akwatia' => "🏘️ **Akwatia** is a vibrant mining town in the Eastern Region of Ghana, known for its diamond mining heritage and rich cultural history. The town has been a significant center for diamond production and continues to be an important economic hub in the region.",
            'ghana' => "🇬🇭 **Ghana** is a West African nation with rich cultural heritage and stable democracy. **Current President**: John Dramani Mahama (inaugurated January 2025).",
            'accra' => "🏙️ **Accra** is the bustling capital city of Ghana, serving as the political, economic, and cultural heart of the nation. The city is experiencing rapid development and modernization while preserving its rich historical heritage.",
            'kumasi' => "👑 **Kumasi** is the historic capital of the Ashanti Kingdom and the second-largest city in Ghana. Known for its rich cultural traditions, vibrant markets, and significant role in Ghana's history.",
            'tamale' => "🌾 **Tamale** is the capital of the Northern Region and one of Ghana's fastest-growing cities. The city is known for its agricultural significance and is becoming an important economic center in northern Ghana."
        ];
        
        return $summaries[strtolower($location)] ?? "📍 **{$location}** is an important location in Ghana with ongoing developments and news worth following.";
    }

    private function generateProfessionalFallback($newsResults, $aiAnalysis)
    {
        $category = $aiAnalysis['category'] ?? 'the news';
        $response = "I've got some interesting developments for you in {$category}. Here's what's happening:\n\n";
        
        foreach ($newsResults as $index => $article) {
            if ($index === 0) {
                $response .= "📰 **{$article['title']}** - {$article['content']}\n\n";
            } else {
                $response .= "Meanwhile, **{$article['title']}** - {$article['content']}\n\n";
            }
        }
        
        $response .= "That's the latest from {$category}. Stay tuned for more updates!";
        
        return $response;
    }
    
    // Removed political leadership methods - now handled by ChatGPT for all countries

    private function storeConversationContext($sessionId, $aiAnalysis, $newsResults)
    {
        $context = [
            'session_id' => $sessionId,
            'timestamp' => time(),
            'intent' => $aiAnalysis['intent'] ?? '',
            'keywords' => $aiAnalysis['keywords'] ?? '',
            'sub_category' => $aiAnalysis['category'] ?? '',
            'shown_article_ids' => array_column($newsResults, 'id'),
            'query_count' => 1
        ];
        
        // Store in Laravel session
        session([$sessionId => $context]);
    }

    private function getConversationContext($sessionId)
    {
        return session($sessionId);
    }

    private function updateConversationContext($sessionId, $additionalNews)
    {
        $context = session($sessionId);
        
        if ($context) {
            $context['shown_article_ids'] = array_merge(
                $context['shown_article_ids'],
                array_column($additionalNews, 'id')
            );
            $context['query_count']++;
            
            // Update the session
            session([$sessionId => $context]);
        }
    }

    /**
     * Save chat session (for FAB chatbot - logged-in users)
     */
    public function saveChatSession(Request $request)
    {
        if (!auth()->check()) {
            return response()->json(['success' => false, 'error' => 'Unauthorized'], 401);
        }

        $request->validate([
            'name' => 'nullable|string|max:255',
            'messages' => 'required|array',
            'messages.*.role' => 'required|in:user,assistant',
            'messages.*.content' => 'required|string',
            'page_url' => 'nullable|string|max:500',
            'page_title' => 'nullable|string|max:255',
        ]);

        $session = ChatSession::create([
            'user_id' => auth()->id(),
            'name' => $request->input('name', 'Chat ' . now()->format('M j, g:i A')),
            'messages' => $request->messages,
            'last_page_url' => $request->page_url,
            'last_page_title' => $request->page_title,
        ]);

        return response()->json(['success' => true, 'session' => $session]);
    }

    /**
     * List chat sessions for current user
     */
    public function listChatSessions(Request $request)
    {
        if (!auth()->check()) {
            return response()->json(['success' => false, 'error' => 'Unauthorized'], 401);
        }

        $sessions = ChatSession::where('user_id', auth()->id())
            ->orderByDesc('updated_at')
            ->limit(50)
            ->get(['id', 'name', 'last_page_title', 'updated_at', 'messages']);

        return response()->json(['success' => true, 'sessions' => $sessions]);
    }

    /**
     * Get a specific chat session to restore
     */
    public function getChatSession($id)
    {
        if (!auth()->check()) {
            return response()->json(['success' => false, 'error' => 'Unauthorized'], 401);
        }

        $session = ChatSession::where('user_id', auth()->id())->findOrFail($id);
        return response()->json(['success' => true, 'session' => $session]);
    }

}
