<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\NotificationCampaign;
use App\Models\UserNotificationSubscription;
use App\Models\NotificationLog;
use App\Services\OneSignalService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class NotificationController extends Controller
{
    protected $oneSignalService;

    public function __construct(OneSignalService $oneSignalService)
    {
        $this->oneSignalService = $oneSignalService;
    }

    /**
     * Display a listing of notification campaigns.
     */
    public function index()
    {
        $campaigns = NotificationCampaign::orderBy('created_at', 'desc')->paginate(20);
        
        return view('admin.notifications.index', compact('campaigns'));
    }

    /**
     * Show the form for creating a new campaign.
     */
    public function create()
    {
        $categories = ['politics', 'business', 'sports', 'technology', 'entertainment', 'health', 'science', 'world', 'local'];
        $locations = ['ghana', 'accra', 'akwatia', 'kumasi', 'tema', 'tamale', 'cape coast', 'sekondi-takoradi'];
        $interests = ['technology', 'entertainment', 'sports', 'politics', 'business', 'health', 'science', 'travel', 'food', 'fashion'];
        
        return view('admin.notifications.create', compact('categories', 'locations', 'interests'));
    }

    /**
     * Store a newly created campaign.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string|max:500',
            'icon' => 'nullable|string|max:255',
            'image_url' => 'nullable|url|max:255',
            'action_url' => 'nullable|url|max:255',
            'action_text' => 'nullable|string|max:100',
            'target_categories' => 'nullable|array',
            'target_locations' => 'nullable|array',
            'target_interests' => 'nullable|array',
            'target_audience' => 'required|in:all,registered,premium',
            'scheduled_at' => 'nullable|date|after:now',
            'expires_at' => 'nullable|date|after:scheduled_at',
            'is_immediate' => 'boolean',
            'status' => 'required|in:draft,scheduled,sent,cancelled',
        ]);

        $data = $request->all();
        
        // Handle image upload
        if ($request->hasFile('image_file')) {
            $path = \App\Services\ImageService::storeUploadedFile($request->file('image_file'), 'notifications', 'public');
            $data['image_url'] = \App\Services\ImageService::getImageUrl($path, 'public');
        }

        // Set immediate flag
        $data['is_immediate'] = $request->boolean('is_immediate');

        // If immediate, set scheduled_at to now
        if ($data['is_immediate']) {
            $data['scheduled_at'] = now();
            $data['status'] = 'scheduled';
        }

        $campaign = NotificationCampaign::create($data);

        // If immediate, send the campaign
        if ($data['is_immediate']) {
            $this->oneSignalService->sendCampaign($campaign);
        }

        return redirect()->route('admin.notifications.index')
            ->with('success', 'Notification campaign created successfully!');
    }

    /**
     * Display the specified campaign.
     */
    public function show(NotificationCampaign $notification)
    {
        $logs = $notification->logs()->with(['user', 'subscription'])->paginate(20);
        
        return view('admin.notifications.show', compact('notification', 'logs'));
    }

    /**
     * Show the form for editing the campaign.
     */
    public function edit(NotificationCampaign $notification)
    {
        $categories = ['politics', 'business', 'sports', 'technology', 'entertainment', 'health', 'science', 'world', 'local'];
        $locations = ['ghana', 'accra', 'akwatia', 'kumasi', 'tema', 'tamale', 'cape coast', 'sekondi-takoradi'];
        $interests = ['technology', 'entertainment', 'sports', 'politics', 'business', 'health', 'science', 'travel', 'food', 'fashion'];
        
        return view('admin.notifications.edit', compact('notification', 'categories', 'locations', 'interests'));
    }

    /**
     * Update the specified campaign.
     */
    public function update(Request $request, NotificationCampaign $notification)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string|max:500',
            'icon' => 'nullable|string|max:255',
            'image_url' => 'nullable|url|max:255',
            'action_url' => 'nullable|url|max:255',
            'action_text' => 'nullable|string|max:100',
            'target_categories' => 'nullable|array',
            'target_locations' => 'nullable|array',
            'target_interests' => 'nullable|array',
            'target_audience' => 'required|in:all,registered,premium',
            'scheduled_at' => 'nullable|date|after:now',
            'expires_at' => 'nullable|date|after:scheduled_at',
            'is_immediate' => 'boolean',
            'status' => 'required|in:draft,scheduled,sent,cancelled',
        ]);

        $data = $request->all();
        
        // Handle image upload
        if ($request->hasFile('image_file')) {
            $path = \App\Services\ImageService::storeUploadedFile($request->file('image_file'), 'notifications', 'public');
            $data['image_url'] = \App\Services\ImageService::getImageUrl($path, 'public');
        }

        // Set immediate flag
        $data['is_immediate'] = $request->boolean('is_immediate');

        // If immediate, set scheduled_at to now
        if ($data['is_immediate']) {
            $data['scheduled_at'] = now();
            $data['status'] = 'scheduled';
        }

        $notification->update($data);

        // If immediate, send the campaign
        if ($data['is_immediate']) {
            $this->oneSignalService->sendCampaign($notification);
        }

        return redirect()->route('admin.notifications.index')
            ->with('success', 'Notification campaign updated successfully!');
    }

    /**
     * Send the specified campaign.
     */
    public function send(NotificationCampaign $notification)
    {
        if ($notification->status !== 'scheduled') {
            return redirect()->back()
                ->with('error', 'Only scheduled campaigns can be sent.');
        }

        $success = $this->oneSignalService->sendCampaign($notification);

        if ($success) {
            return redirect()->back()
                ->with('success', 'Campaign sent successfully!');
        } else {
            return redirect()->back()
                ->with('error', 'Failed to send campaign. Please try again.');
        }
    }

    /**
     * Cancel the specified campaign.
     */
    public function cancel(NotificationCampaign $notification)
    {
        if ($notification->status === 'sent') {
            return redirect()->back()
                ->with('error', 'Cannot cancel a campaign that has already been sent.');
        }

        $notification->update(['status' => 'cancelled']);

        return redirect()->back()
            ->with('success', 'Campaign cancelled successfully!');
    }

    /**
     * Remove the specified campaign.
     */
    public function destroy(NotificationCampaign $notification)
    {
        if ($notification->status === 'sent') {
            return redirect()->back()
                ->with('error', 'Cannot delete a campaign that has already been sent.');
        }

        $notification->delete();

        return redirect()->route('admin.notifications.index')
            ->with('success', 'Campaign deleted successfully!');
    }

    /**
     * Display notification analytics.
     */
    public function analytics()
    {
        $stats = [
            'total_campaigns' => NotificationCampaign::count(),
            'sent_campaigns' => NotificationCampaign::sent()->count(),
            'scheduled_campaigns' => NotificationCampaign::scheduled()->count(),
            'total_subscriptions' => UserNotificationSubscription::active()->count(),
            'total_sent' => NotificationLog::sum('sent_count'),
            'total_delivered' => NotificationLog::delivered()->count(),
            'total_clicked' => NotificationLog::clicked()->count(),
            'total_converted' => NotificationLog::converted()->count(),
        ];

        // Recent campaigns
        $recentCampaigns = NotificationCampaign::sent()
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // Top performing campaigns
        $topCampaigns = NotificationCampaign::sent()
            ->orderBy('clicked_count', 'desc')
            ->limit(5)
            ->get();

        // Daily stats for the last 30 days
        $dailyStats = NotificationLog::selectRaw('DATE(sent_at) as date, COUNT(*) as sent, SUM(CASE WHEN status = "delivered" THEN 1 ELSE 0 END) as delivered, SUM(CASE WHEN status = "clicked" THEN 1 ELSE 0 END) as clicked')
            ->where('sent_at', '>=', now()->subDays(30))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        return view('admin.notifications.analytics', compact('stats', 'recentCampaigns', 'topCampaigns', 'dailyStats'));
    }

    /**
     * Display subscription management.
     */
    public function subscriptions()
    {
        $subscriptions = UserNotificationSubscription::with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(50);

        $stats = [
            'total_subscriptions' => UserNotificationSubscription::count(),
            'active_subscriptions' => UserNotificationSubscription::active()->count(),
            'web_subscriptions' => UserNotificationSubscription::where('device_type', 'web')->count(),
            'mobile_subscriptions' => UserNotificationSubscription::whereIn('device_type', ['android', 'ios'])->count(),
        ];

        return view('admin.notifications.subscriptions', compact('subscriptions', 'stats'));
    }

    /**
     * Send test notification.
     */
    public function sendTest(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string|max:500',
            'test_player_id' => 'required|string',
        ]);

        $notificationData = [
            'title' => $request->title,
            'message' => $request->message,
            'data' => [
                'type' => 'test_notification',
            ],
        ];

        $success = $this->oneSignalService->sendToPlayers([$request->test_player_id], $notificationData);

        if ($success) {
            return redirect()->back()
                ->with('success', 'Test notification sent successfully!');
        } else {
            return redirect()->back()
                ->with('error', 'Failed to send test notification.');
        }
    }
}