<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Symfony\Component\Process\Process;

class SyncPeaceFm extends Command
{
    protected $signature = 'election:sync-peacefm
                            {slug? : Slug to sync (2000, 2000a, 2004, 2008, 2008a, 2016, 2020, 2024)}
                            {--all : Scrape and import all years}
                            {--scrape : Run scraper before import (default when no JSON exists)}
                            {--no-scrape : Only import; do not run scraper}';

    protected $description = 'Scrape Peace FM and import: one command to populate all (or one) election. Use --all for every year.';

    private const SLUGS = ['2000', '2000a', '2004', '2008', '2008a', '2016', '2020', '2024'];

    public function handle(): int
    {
        $slugArg = $this->argument('slug');
        $all = $this->option('all');
        $runScraper = $this->option('scrape') || (! $this->option('no-scrape') && $this->shouldScrape($slugArg, $all));

        if ($all) {
            $slugs = self::SLUGS;
        } elseif ($slugArg && in_array($slugArg, self::SLUGS, true)) {
            $slugs = [$slugArg];
        } else {
            $this->error('Provide a slug (e.g. 2016, 2020) or use --all.');
            $this->line('Slugs: ' . implode(', ', self::SLUGS));
            return self::FAILURE;
        }

        if ($runScraper) {
            $scrapeArg = $all ? 'all' : $slugs[0];
            $this->info('Running Peace FM scraper: ' . $scrapeArg);
            $process = new Process(
                ['node', 'scripts/scrape-peacefm.mjs', $scrapeArg],
                base_path(),
                null,
                null,
                300
            );
            $process->run(function ($type, $buffer) {
                $this->line($buffer);
            });
            if (! $process->isSuccessful()) {
                $this->error('Scraper failed.');
                return self::FAILURE;
            }
        }

        $this->info('Importing...');
        if ($all) {
            $this->call('election:import-peacefm', ['--all' => true]);
        } else {
            $ok = $this->call('election:import-peacefm', ['slug' => $slugs[0]]) === self::SUCCESS;
            if (! $ok) {
                return self::FAILURE;
            }
        }

        $this->info('Sync complete.');
        return self::SUCCESS;
    }

    private function shouldScrape(?string $slug, bool $all): bool
    {
        if ($all) {
            $missing = 0;
            foreach (self::SLUGS as $s) {
                if (! is_file(base_path("storage/app/peacefm_{$s}_presidential.json"))) {
                    $missing++;
                }
            }
            return $missing > 0;
        }
        if (! $slug) {
            return false;
        }
        return ! is_file(base_path("storage/app/peacefm_{$slug}_presidential.json"));
    }
}
