<?php

namespace App\Console\Commands;

use App\Models\Election\Election;
use App\Models\Election\ElectionScrapedData;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class ImportPeaceFm2020 extends Command
{
    protected $signature = 'election:import-peacefm {slug? : Peace FM slug (2000, 2000a, 2004, 2008, 2008a, 2016, 2020, 2024) or path to JSON} {--all : Import all scraped JSON files}';
    protected $description = 'Import Peace FM presidential data from JSON. Run: node scripts/scrape-peacefm.mjs <slug|all> first.';

    private const SLUGS = ['2000', '2000a', '2004', '2008', '2008a', '2016', '2020', '2024'];

    public function handle(): int
    {
        if ($this->option('all')) {
            foreach (self::SLUGS as $slug) {
                $path = base_path("storage/app/peacefm_{$slug}_presidential.json");
                if (is_file($path)) {
                    $this->info("Importing {$slug}...");
                    $this->importFile($path, $slug);
                } else {
                    $this->warn("Skip {$slug}: file not found.");
                }
            }
            $this->info('Import complete.');
            return self::SUCCESS;
        }

        $slugOrPath = $this->argument('slug') ?? '2020';
        $path = $this->resolvePath($slugOrPath);
        if (! is_file($path)) {
            $this->error("File not found: {$path}");
            $this->line('Slugs: ' . implode(', ', self::SLUGS));
            $this->line('Or run: node scripts/scrape-peacefm.mjs <slug>');
            return self::FAILURE;
        }

        $slug = $this->pathToSlug($path) ?? $slugOrPath;
        return $this->importFile($path, $slug) ? self::SUCCESS : self::FAILURE;
    }

    private function resolvePath(string $slugOrPath): string
    {
        if (str_contains($slugOrPath, '.json') || str_contains($slugOrPath, '/') || str_contains($slugOrPath, '\\')) {
            return base_path($slugOrPath);
        }
        return base_path("storage/app/peacefm_{$slugOrPath}_presidential.json");
    }

    private function pathToSlug(string $path): ?string
    {
        if (preg_match('/peacefm_(\d+a?)_presidential\.json$/', $path, $m)) {
            return $m[1];
        }
        return null;
    }

    private function slugToElection(string $slug): ?Election
    {
        $runoff = str_ends_with($slug, 'a');
        $year = (int) rtrim($slug, 'a');
        $type = $runoff ? 'presidential_runoff' : 'presidential';
        return Election::where('year', $year)->where('type', $type)->first();
    }

    private function importFile(string $path, string $slug): bool
    {
        $json = json_decode(file_get_contents($path), true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->error('Invalid JSON: ' . json_last_error_msg());
            return false;
        }

        $election = $this->slugToElection($slug);
        if (! $election) {
            $this->error("Election not found for slug: {$slug}. Run php artisan election:clear first.");
            return false;
        }

        $year = $election->year;
        $presidentUrl = $json['pages']['president']['url'] ?? 'https://www.peacefmonline.com/pages/' . $slug . '/president';
        $swingUrl = $json['pages']['swing']['url'] ?? $presidentUrl . ($slug === '2024' ? '/analyses/swing-constituencies' : '/swing');
        $marginalUrl = $json['pages']['marginal']['url'] ?? $presidentUrl . ($slug === '2024' ? '/analyses/marginal' : '/marginal');
        $regionalUrl = $json['pages']['regional_analysis']['url'] ?? ($slug === '2024'
            ? 'https://www.peacefmonline.com/elections/2024/president/analyses/region-results'
            : 'https://www.peacefmonline.com/pages/' . $slug . '/analysis/president');

        if (! empty($json['national'])) {
            $nationalData = $json['national']['national'] ?? $json['national'];
            $regionalVotesFromJson = $json['national']['regional_votes'] ?? [];

            $tables = $json['national']['raw_tables'] ?? $json['pages']['president']['tables'] ?? [];
            $derived = ! empty($tables) ? $this->deriveNationalFromPresidentTable($tables) : [];
            if (! empty($derived)) {
                if (empty($nationalData['npp'])) {
                    $nationalData['npp'] = $derived['npp'] ?? null;
                }
                if (empty($nationalData['ndc'])) {
                    $nationalData['ndc'] = $derived['ndc'] ?? null;
                }
                if (empty($nationalData['npp']) && empty($nationalData['ndc'])) {
                    $nationalData = $derived;
                }
                if ($derived['npp'] ?? $derived['ndc'] ?? null) {
                    $this->line('Derived national NPP/NDC from president table where missing.');
                }
            }

            $nationalPayload = [
                'source_url' => $presidentUrl,
                'scraped_at' => $json['scraped_at'] ?? now()->toIso8601String(),
                'national' => $nationalData,
                'regional_votes' => $regionalVotesFromJson,
            ];
            ElectionScrapedData::updateOrCreate(
                ['election_id' => $election->id, 'data_type' => 'national_summary'],
                [
                    'payload' => $nationalPayload,
                    'source_url' => $presidentUrl,
                    'scraped_at' => now(),
                ]
            );
            $this->line('Imported national_summary (NPP/NDC: ' . (isset($nationalPayload['national']['npp']) ? 'yes' : 'no') . ', regional_votes: ' . count($nationalPayload['regional_votes']) . ')');
        }

        if (! empty($json['swing'])) {
            $swingPayload = [
                'source_url' => $swingUrl,
                'scraped_at' => $json['scraped_at'] ?? now()->toIso8601String(),
                'swing_list' => $json['swing']['swing_list'] ?? [],
                'by_region' => $json['swing']['by_region'] ?? [],
            ];
            if ($year === 2024 && isset($json['swing']['by_region'])) {
                $swingPayload['by_region'] = $json['swing']['by_region'];
            }
            ElectionScrapedData::updateOrCreate(
                ['election_id' => $election->id, 'data_type' => 'swing_constituencies'],
                [
                    'payload' => $swingPayload,
                    'source_url' => $swingUrl,
                    'scraped_at' => now(),
                ]
            );
            $swingCount = count($swingPayload['swing_list']) ?: count($swingPayload['by_region'] ?? []);
            $this->line('Imported swing_constituencies: ' . $swingCount . ' items');
        }

        if (! empty($json['marginal'])) {
            $marginalPayload = [
                'source_url' => $marginalUrl,
                'scraped_at' => $json['scraped_at'] ?? now()->toIso8601String(),
                'marginal_list' => $json['marginal']['marginal_list'] ?? [],
            ];
            ElectionScrapedData::updateOrCreate(
                ['election_id' => $election->id, 'data_type' => 'marginal_wins'],
                [
                    'payload' => $marginalPayload,
                    'source_url' => $marginalUrl,
                    'scraped_at' => now(),
                ]
            );
            $this->line('Imported marginal_wins: ' . count($marginalPayload['marginal_list']) . ' items');
        }

        if (! empty($json['regional_analysis'])) {
            $regionalPayload = [
                'source_url' => $regionalUrl,
                'scraped_at' => $json['scraped_at'] ?? now()->toIso8601String(),
                'regional_comparison' => $json['regional_analysis']['regional_comparison'] ?? [],
            ];
            ElectionScrapedData::updateOrCreate(
                ['election_id' => $election->id, 'data_type' => 'regional_analysis'],
                [
                    'payload' => $regionalPayload,
                    'source_url' => $regionalUrl,
                    'scraped_at' => now(),
                ]
            );
            $this->line('Imported regional_analysis: ' . count($regionalPayload['regional_comparison']) . ' regions');
        }

        if ($year === 2024 && ! empty($json['region_turnout'])) {
            $base2024Url = 'https://www.peacefmonline.com/elections/2024/president';
            $turnoutPayload = [
                'source_url' => $json['pages']['region_turnout']['url'] ?? $base2024Url . '/analyses/region-turnout',
                'scraped_at' => $json['scraped_at'] ?? now()->toIso8601String(),
                'region_turnout' => $json['region_turnout'],
            ];
            ElectionScrapedData::updateOrCreate(
                ['election_id' => $election->id, 'data_type' => 'region_turnout'],
                [
                    'payload' => $turnoutPayload,
                    'source_url' => $turnoutPayload['source_url'],
                    'scraped_at' => now(),
                ]
            );
            $this->line('Imported region_turnout.');
        }

        if ($year === 2024 && ! empty($json['rejected_ballots'])) {
            $rejPayload = [
                'source_url' => $json['pages']['rejected_ballots']['url'] ?? 'https://www.peacefmonline.com/elections/2024/president/analyses/region-rejected-ballots',
                'scraped_at' => $json['scraped_at'] ?? now()->toIso8601String(),
                'rejected_ballots' => $json['rejected_ballots'],
            ];
            ElectionScrapedData::updateOrCreate(
                ['election_id' => $election->id, 'data_type' => 'rejected_ballots'],
                [
                    'payload' => $rejPayload,
                    'source_url' => $rejPayload['source_url'],
                    'scraped_at' => now(),
                ]
            );
            $this->line('Imported rejected_ballots.');
        }

        $this->info('Peace FM ' . $year . ' import complete.');
        return true;
    }

    private function syncRegionalResultsFromScraped(Election $election, array $regionalVotes): void
    {
        $nppId = DB::table('parties')->where('short_name', 'NPP')->value('id');
        $ndcId = DB::table('parties')->where('short_name', 'NDC')->value('id');
        $nppCandidateId = DB::table('candidates')->where('name', 'Nana Akufo-Addo')->value('id');
        $ndcCandidateId = DB::table('candidates')->where('name', 'John Dramani Mahama')->value('id');
        $regionNameToId = DB::table('regions')->pluck('id', 'name')->toArray();
        foreach ($regionalVotes as $r) {
            $name = $r['region'] ?? '';
            $regionId = $regionNameToId[$name] ?? null;
            if (! $regionId || (($r['npp_votes'] ?? 0) + ($r['ndc_votes'] ?? 0)) <= 0) {
                continue;
            }
            $total = (int) (($r['npp_votes'] ?? 0) + ($r['ndc_votes'] ?? 0));
            $nppV = (int) ($r['npp_votes'] ?? 0);
            $ndcV = (int) ($r['ndc_votes'] ?? 0);
            $nppPct = (float) ($r['npp_pct'] ?? ($total ? round(100 * $nppV / $total, 2) : 0));
            $ndcPct = (float) ($r['ndc_pct'] ?? ($total ? round(100 * $ndcV / $total, 2) : 0));
            $winner = $nppV >= $ndcV ? 'NPP' : 'NDC';
            DB::table('election_results_regional')->updateOrInsert(
                ['election_id' => $election->id, 'region_id' => $regionId, 'party_id' => $nppId],
                ['candidate_id' => $nppCandidateId, 'votes' => $nppV, 'percentage' => $nppPct, 'total_votes_in_region' => $total, 'is_region_winner' => $winner === 'NPP', 'updated_at' => now()]
            );
            DB::table('election_results_regional')->updateOrInsert(
                ['election_id' => $election->id, 'region_id' => $regionId, 'party_id' => $ndcId],
                ['candidate_id' => $ndcCandidateId, 'votes' => $ndcV, 'percentage' => $ndcPct, 'total_votes_in_region' => $total, 'is_region_winner' => $winner === 'NDC', 'updated_at' => now()]
            );
        }
    }

    /**
     * Derive national NPP/NDC from president page table (Candidate, Votes, Percent).
     * 2024 table: Mahama → NDC, Bawumia → NPP.
     */
    private function deriveNationalFromPresidentTable(array $tables): array
    {
        $national = ['npp' => null, 'ndc' => null];
        foreach ($tables as $t) {
            $rows = $t['rows'] ?? [];
            if (count($rows) < 2) {
                continue;
            }
            $header = array_map('strtolower', array_map('trim', $rows[0]));
            $votesCol = null;
            $pctCol = null;
            foreach ($header as $i => $h) {
                if (str_contains($h, 'vote')) {
                    $votesCol = $i;
                }
                if (str_contains($h, 'percent') || $h === '%') {
                    $pctCol = $i;
                }
            }
            if ($votesCol === null) {
                $votesCol = 2;
            }
            if ($pctCol === null) {
                $pctCol = 3;
            }
            for ($i = 1; $i < count($rows); $i++) {
                $row = $rows[$i];
                $candidate = (string) ($row[0] ?? '');
                $votesStr = (string) (isset($row[$votesCol]) ? $row[$votesCol] : ($row[2] ?? ''));
                $pctStr = (string) (isset($row[$pctCol]) ? $row[$pctCol] : ($row[3] ?? ''));
                $votes = (int) str_replace(',', '', $votesStr);
                $pct = (float) str_replace('%', '', $pctStr);
                if ($votes <= 0) {
                    continue;
                }
                $lower = strtolower($candidate);
                if (str_contains($lower, 'mahama') || str_contains($lower, 'john dramani')) {
                    $national['ndc'] = ['votes' => $votes, 'percentage' => $pct];
                } elseif (str_contains($lower, 'bawumia') || str_contains($lower, 'mahamudu')) {
                    $national['npp'] = ['votes' => $votes, 'percentage' => $pct];
                }
            }
            if ($national['npp'] !== null || $national['ndc'] !== null) {
                return $national;
            }
        }
        return [];
    }
}
