import axios, { AxiosInstance, AxiosResponse } from 'axios';
import { ChatResponse, NewsArticle, ApiResponse } from '../types';
import { API_CONFIG } from '../config/api';

class ApiService {
  private api: AxiosInstance;
  private baseURL: string;

  constructor() {
    this.baseURL = API_CONFIG.BASE_URL;
    
    this.api = axios.create({
      baseURL: this.baseURL,
      timeout: API_CONFIG.TIMEOUT,
      headers: API_CONFIG.HEADERS,
    });

    // Add request interceptor for CSRF token
    this.api.interceptors.request.use(
      (config) => {
        // Add any auth tokens here if needed
        return config;
      },
      (error) => {
        return Promise.reject(error);
      }
    );

    // Add response interceptor for error handling
    this.api.interceptors.response.use(
      (response: AxiosResponse) => {
        return response;
      },
      (error) => {
        console.error('API Error:', error.response?.data || error.message);
        return Promise.reject(error);
      }
    );
  }

  // Chat API
  async sendChatMessage(query: string, sessionId: string = 'mobile_session'): Promise<ChatResponse> {
    try {
      const response = await this.api.post('/api/chat', {
        query,
        session_id: sessionId,
      });

      return response.data;
    } catch (error) {
      console.error('Chat API Error:', error);
      throw new Error('Failed to send chat message');
    }
  }

  // News API
  async getLatestNews(limit: number = 20): Promise<NewsArticle[]> {
    try {
      console.log('API: Fetching latest news with limit:', limit);
      console.log('API: Base URL:', this.baseURL);
      console.log('API: Full URL:', `${this.baseURL}/api/news?limit=${limit}`);
      
      const response = await this.api.get('/api/news', {
        params: { limit }
      });

      console.log('API: News response:', response.data);
      console.log('API: Articles count:', response.data.data?.length || 0);
      
      if (!response.data.success) {
        throw new Error(response.data.error || 'API returned unsuccessful response');
      }
      
      return response.data.data || [];
    } catch (error) {
      console.error('News API Error:', error);
      if (error.response) {
        console.error('Error response:', error.response.data);
        console.error('Error status:', error.response.status);
      }
      throw new Error('Failed to fetch news: ' + (error.message || 'Unknown error'));
    }
  }

  // Sample news data for development
  private getSampleNews(): NewsArticle[] {
    return [
      {
        id: '1',
        post_title: 'Sample News Article 1',
        post_content: 'This is sample content for development purposes.',
        post_excerpt: 'Sample excerpt for the news article.',
        post_url: 'https://example.com',
        post_image: 'https://via.placeholder.com/400x200/0a57d4/ffffff?text=News+1',
        post_category: 'Technology',
        post_source: 'Sample Source',
        date_added: new Date().toISOString(),
        views: 150
      },
      {
        id: '2',
        post_title: 'Sample News Article 2',
        post_content: 'This is another sample content for development purposes.',
        post_excerpt: 'Another sample excerpt for the news article.',
        post_url: 'https://example.com',
        post_image: 'https://via.placeholder.com/400x200/2d72f0/ffffff?text=News+2',
        post_category: 'Business',
        post_source: 'Sample Source',
        date_added: new Date().toISOString(),
        views: 200
      },
      {
        id: '3',
        post_title: 'Sample News Article 3',
        post_content: 'This is a third sample content for development purposes.',
        post_excerpt: 'A third sample excerpt for the news article.',
        post_url: 'https://example.com',
        post_image: 'https://via.placeholder.com/400x200/5a95ff/ffffff?text=News+3',
        post_category: 'Politics',
        post_source: 'Sample Source',
        date_added: new Date().toISOString(),
        views: 300
      }
    ];
  }

  async getNewsByCategory(category: string, limit: number = 20): Promise<NewsArticle[]> {
    try {
      console.log('API: Fetching news for category:', category);
      console.log('API: Full URL:', `${this.baseURL}/api/news?category=${category}&limit=${limit}`);
      
      const response = await this.api.get('/api/news', {
        params: { category, limit }
      });

      console.log('API: Category news response:', response.data);
      
      if (!response.data.success) {
        throw new Error(response.data.error || 'API returned unsuccessful response');
      }

      return response.data.data || [];
    } catch (error) {
      console.error('Category News API Error:', error);
      if (error.response) {
        console.error('Error response:', error.response.data);
        console.error('Error status:', error.response.status);
      }
      throw new Error('Failed to fetch category news: ' + (error.message || 'Unknown error'));
    }
  }

  async searchNews(query: string, limit: number = 20): Promise<NewsArticle[]> {
    try {
      const response = await this.api.get('/api/news', {
        params: { q: query, limit }
      });

      return response.data.data || [];
    } catch (error) {
      console.error('Search API Error:', error);
      throw new Error('Failed to search news');
    }
  }

  async getNewsCategories(): Promise<string[]> {
    try {
      console.log('API: Fetching news categories...');
      console.log('API: Full URL:', `${this.baseURL}/api/news/categories`);
      
      const response = await this.api.get('/api/news/categories');
      console.log('API: Categories response:', response.data);
      
      if (!response.data.success) {
        throw new Error(response.data.error || 'API returned unsuccessful response');
      }
      
      return response.data.data || [];
    } catch (error) {
      console.error('Categories API Error:', error);
      if (error.response) {
        console.error('Error response:', error.response.data);
        console.error('Error status:', error.response.status);
      }
      // Return fallback categories instead of throwing error
      console.log('API: Using fallback categories');
      return ['Politics', 'Business', 'Entertainment', 'Sports', 'Technology', 'Health', 'International'];
    }
  }

  // Authentication API (if needed)
  async login(email: string, password: string): Promise<ApiResponse<any>> {
    try {
      const response = await this.api.post('/api/auth/login', {
        email,
        password,
      });

      return response.data;
    } catch (error) {
      console.error('Login API Error:', error);
      throw new Error('Failed to login');
    }
  }

  async register(name: string, email: string, password: string): Promise<ApiResponse<any>> {
    try {
      const response = await this.api.post('/api/auth/register', {
        name,
        email,
        password,
        password_confirmation: password,
      });

      return response.data;
    } catch (error) {
      console.error('Register API Error:', error);
      throw new Error('Failed to register');
    }
  }

  // Deep Dive API
  async getArticle(articleId: string): Promise<ApiResponse<NewsArticle>> {
    try {
      const response = await this.api.get(`/api/news/deep-dive/${articleId}`);
      return response.data;
    } catch (error) {
      console.error('Get Article API Error:', error);
      throw new Error('Failed to fetch article');
    }
  }

  async askQuestion(articleId: string, question: string): Promise<ApiResponse<{ response: string }>> {
    try {
      const response = await this.api.post(`/api/news/deep-dive/${articleId}/chat`, {
        message: question
      });
      return response.data;
    } catch (error) {
      console.error('Ask Question API Error:', error);
      throw new Error('Failed to ask question');
    }
  }

  // Utility method to update base URL
  updateBaseURL(newBaseURL: string) {
    this.baseURL = newBaseURL;
    this.api.defaults.baseURL = newBaseURL;
  }
}

// Export singleton instance
export const apiService = new ApiService();
export default apiService;
