import React, { useEffect, useState } from 'react';
import { 
  View, 
  StyleSheet, 
  FlatList, 
  RefreshControl, 
  TouchableOpacity, 
  Image, 
  StatusBar,
  ScrollView,
  Animated
} from 'react-native';
import { useDispatch, useSelector } from 'react-redux';
import { Text, ActivityIndicator } from 'react-native-paper';
import { useNavigation } from '@react-navigation/native';
import { RootState, AppDispatch } from '../store';
import { fetchLatestNews, fetchNewsByCategory, setSelectedCategory, fetchCategories } from '../store/slices/newsSlice';
import { NewsArticle } from '../types';

// Utility function to format relative dates
const formatRelativeDate = (dateString: string): string => {
  const now = new Date();
  const date = new Date(dateString);
  const diffInMs = now.getTime() - date.getTime();
  const diffInDays = Math.floor(diffInMs / (1000 * 60 * 60 * 24));
  const diffInHours = Math.floor(diffInMs / (1000 * 60 * 60));
  const diffInMinutes = Math.floor(diffInMs / (1000 * 60));

  if (diffInDays > 0) {
    return `${diffInDays} day${diffInDays > 1 ? 's' : ''} ago`;
  } else if (diffInHours > 0) {
    return `${diffInHours} hour${diffInHours > 1 ? 's' : ''} ago`;
  } else if (diffInMinutes > 0) {
    return `${diffInMinutes} minute${diffInMinutes > 1 ? 's' : ''} ago`;
  } else {
    return 'Just now';
  }
};

const NewsDiscoveryScreen: React.FC = () => {
  const dispatch = useDispatch<AppDispatch>();
  const navigation = useNavigation();
  const { articles, categories, isLoading, selectedCategory, error } = useSelector((state: RootState) => state.news);
  const [activeTab, setActiveTab] = useState('For You');
  const fadeAnim = useState(new Animated.Value(0))[0];

  const tabs = ['For You', 'Top Stories', 'Tech & Science', 'Finance'];

  useEffect(() => {
    console.log('NewsDiscoveryScreen: Fetching news and categories...');
    dispatch(fetchLatestNews(20));
    dispatch(fetchCategories());
    
    // Fade in animation
    Animated.timing(fadeAnim, {
      toValue: 1,
      duration: 800,
      useNativeDriver: true,
    }).start();
  }, [dispatch, fadeAnim]);

  // Debug logging
  useEffect(() => {
    console.log('NewsDiscoveryScreen: Articles updated:', articles.length);
    console.log('NewsDiscoveryScreen: Categories:', categories);
    console.log('NewsDiscoveryScreen: Loading:', isLoading);
    console.log('NewsDiscoveryScreen: Error:', error);
  }, [articles, categories, isLoading, error]);

  const handleRefresh = () => {
    if (selectedCategory) {
      dispatch(fetchNewsByCategory({ category: selectedCategory }));
    } else {
      dispatch(fetchLatestNews(20));
    }
  };

  const handleCategoryPress = (category: string) => {
    if (selectedCategory === category) {
      dispatch(setSelectedCategory(null));
      dispatch(fetchLatestNews(20));
    } else {
      dispatch(setSelectedCategory(category));
      dispatch(fetchNewsByCategory({ category }));
    }
  };

  const handleArticlePress = (article: NewsArticle) => {
    // Navigate to deep dive screen
    (navigation as any).navigate('Article', { article });
  };

  const handleTabPress = (tab: string) => {
    setActiveTab(tab);
    // Add haptic feedback or animation here
  };

  const testApiConnection = async () => {
    try {
      console.log('Testing API connection...');
      const response = await fetch('http://192.168.100.227/mews/public/api/news');
      const data = await response.json();
      console.log('API Test Response:', data);
      alert(`API Test: Found ${data.count || 0} articles`);
    } catch (error) {
      console.error('API Test Error:', error);
      alert('API Test Failed: ' + error.message);
    }
  };

  const renderArticle = ({ item }: { item: NewsArticle }) => {
    const imageUrl = item.post_image && item.post_image !== 'N/A' && item.post_image.trim() !== ''
      ? (item.post_image.startsWith('http') 
          ? item.post_image 
          : `http://192.168.100.227/mews/public${item.post_image}`)
      : 'https://via.placeholder.com/300x180?text=No+Image';

    const relativeDate = formatRelativeDate(item.date_added);
    const source = item.post_source || 'Unknown Source';

    return (
      <TouchableOpacity 
        style={styles.articleCard} 
        activeOpacity={0.8}
        onPress={() => handleArticlePress(item)}
      >
        <View style={styles.articleImageContainer}>
          <Image 
            source={{ uri: imageUrl }} 
            style={styles.articleImage}
            resizeMode="cover"
          />
          <View style={styles.articleCategoryBadge}>
            <Text style={styles.articleCategoryText}>{item.post_category}</Text>
          </View>
        </View>
        <View style={styles.articleContent}>
          <Text style={styles.articleTitle} numberOfLines={2}>
            {item.post_title}
          </Text>
          <View style={styles.articleMeta}>
            <View style={styles.articleMetaLeft}>
              <Text style={styles.articleSource}>{source}</Text>
              <Text style={styles.articleDate}>{relativeDate}</Text>
            </View>
            <View style={styles.articleActions}>
              <TouchableOpacity style={styles.actionButton}>
                <Text style={styles.actionIcon}>♡</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.actionButton}>
                <Text style={styles.actionIcon}>⋯</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </TouchableOpacity>
    );
  };

  const renderCategory = (category: string) => (
    <TouchableOpacity
      key={category}
      style={[
        styles.categoryButton,
        selectedCategory === category && styles.selectedCategoryButton
      ]}
      onPress={() => handleCategoryPress(category)}
      activeOpacity={0.7}
    >
      <Text style={[
        styles.categoryButtonText,
        selectedCategory === category && styles.selectedCategoryButtonText
      ]}>
        {category}
      </Text>
    </TouchableOpacity>
  );

  if (isLoading && articles.length === 0) {
    return (
      <View style={styles.loadingContainer}>
        <ActivityIndicator size="large" color="#0a57d4" />
        <Text style={styles.loadingText}>Loading news...</Text>
      </View>
    );
  }

  return (
    <Animated.View style={[styles.container, { opacity: fadeAnim }]}>
      <StatusBar barStyle="dark-content" backgroundColor="#ffffff" />
      
      {/* Header */}
      <View style={styles.header}>
        <TouchableOpacity style={styles.backButton}>
          <Text style={styles.backIcon}>←</Text>
        </TouchableOpacity>
        <Text style={styles.headerTitle}>Discover</Text>
        <TouchableOpacity style={styles.actionButton} onPress={testApiConnection}>
          <Text style={styles.actionIcon}>🔧</Text>
        </TouchableOpacity>
      </View>

      {/* Tab Navigation */}
      <ScrollView 
        horizontal 
        showsHorizontalScrollIndicator={false}
        style={styles.tabContainer}
        contentContainerStyle={styles.tabContent}
      >
        {tabs.map((tab) => (
          <TouchableOpacity
            key={tab}
            style={[
              styles.tab,
              activeTab === tab && styles.activeTab
            ]}
            onPress={() => handleTabPress(tab)}
            activeOpacity={0.7}
          >
            <Text style={[
              styles.tabText,
              activeTab === tab && styles.activeTabText
            ]}>
              {tab}
            </Text>
          </TouchableOpacity>
        ))}
      </ScrollView>

      {/* News Articles */}
      <FlatList
        data={articles}
        renderItem={renderArticle}
        keyExtractor={(item) => item.id || item.post_id || Math.random().toString()}
        refreshControl={
          <RefreshControl
            refreshing={isLoading}
            onRefresh={handleRefresh}
            colors={['#0a57d4']}
            tintColor="#0a57d4"
          />
        }
        contentContainerStyle={styles.articlesList}
        showsVerticalScrollIndicator={false}
      />
    </Animated.View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#ffffff',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#ffffff',
  },
  loadingText: {
    marginTop: 16,
    fontSize: 16,
    color: '#666',
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    paddingTop: 50,
    paddingBottom: 20,
    backgroundColor: '#ffffff',
  },
  backButton: {
    width: 40,
    height: 40,
    alignItems: 'center',
    justifyContent: 'center',
  },
  backIcon: {
    fontSize: 24,
    color: '#666',
  },
  headerTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#000',
  },
  actionButton: {
    width: 40,
    height: 40,
    alignItems: 'center',
    justifyContent: 'center',
  },
  actionIcon: {
    fontSize: 20,
    color: '#666',
  },
  tabContainer: {
    backgroundColor: '#ffffff',
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  tabContent: {
    paddingHorizontal: 20,
    paddingVertical: 16,
  },
  tab: {
    paddingHorizontal: 16,
    paddingVertical: 8,
    marginRight: 20,
    borderRadius: 20,
  },
  activeTab: {
    backgroundColor: '#0a57d4',
  },
  tabText: {
    fontSize: 16,
    color: '#999',
    fontWeight: '500',
  },
  activeTabText: {
    color: '#ffffff',
    fontWeight: '600',
  },
  articlesList: {
    padding: 20,
  },
  articleCard: {
    backgroundColor: '#ffffff',
    borderRadius: 20,
    marginBottom: 24,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 6 },
    shadowOpacity: 0.08,
    shadowRadius: 12,
    elevation: 6,
    overflow: 'hidden',
  },
  articleImageContainer: {
    position: 'relative',
    height: 220,
  },
  articleImage: {
    width: '100%',
    height: '100%',
  },
  articleCategoryBadge: {
    position: 'absolute',
    top: 16,
    left: 16,
    backgroundColor: 'rgba(10, 87, 212, 0.9)',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 12,
  },
  articleCategoryText: {
    color: '#ffffff',
    fontSize: 12,
    fontWeight: '600',
  },
  articleContent: {
    padding: 20,
  },
  articleTitle: {
    fontSize: 18,
    fontWeight: '700',
    color: '#1a1a1a',
    marginBottom: 16,
    lineHeight: 26,
  },
  articleMeta: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  articleMetaLeft: {
    flex: 1,
  },
  articleSource: {
    fontSize: 14,
    color: '#0a57d4',
    fontWeight: '600',
    marginBottom: 4,
  },
  articleDate: {
    fontSize: 13,
    color: '#8e8e93',
    fontWeight: '500',
  },
  articleActions: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 12,
  },
});

export default NewsDiscoveryScreen;
