import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  StyleSheet,
  TouchableOpacity,
  Text,
  Dimensions,
  Animated,
  StatusBar,
  SafeAreaView,
} from 'react-native';
import { useDispatch, useSelector } from 'react-redux';
import { useNavigation } from '@react-navigation/native';
import { RootState, AppDispatch } from '../store';
import { useTheme } from '../contexts/ThemeContext';
import ChatScreen from './ChatScreen';
import NewsDiscoveryScreen from './NewsDiscoveryScreen';

const { width: screenWidth, height: screenHeight } = Dimensions.get('window');

const MainScreen: React.FC = () => {
  const navigation = useNavigation();
  const [showDiscover, setShowDiscover] = useState(false);
  const pulseAnim = useRef(new Animated.Value(1)).current;
  const slideAnim = useRef(new Animated.Value(0)).current;
  const fadeAnim = useRef(new Animated.Value(1)).current;
  const { theme, toggleTheme, isDark } = useTheme();
  
  const dynamicStyles = createDynamicStyles(isDark);

  // Pulsing animation for profile button
  useEffect(() => {
    const pulse = () => {
      Animated.sequence([
        Animated.timing(pulseAnim, {
          toValue: 1.1,
          duration: 1000,
          useNativeDriver: true,
        }),
        Animated.timing(pulseAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        }),
      ]).start(() => pulse());
    };
    pulse();
  }, [pulseAnim]);

  // Slide animation for discover toggle
  useEffect(() => {
    Animated.parallel([
      Animated.timing(slideAnim, {
        toValue: showDiscover ? 1 : 0,
        duration: 300,
        useNativeDriver: true,
      }),
      Animated.timing(fadeAnim, {
        toValue: showDiscover ? 0.3 : 1,
        duration: 300,
        useNativeDriver: true,
      }),
    ]).start();
  }, [showDiscover, slideAnim, fadeAnim]);

  return (
    <SafeAreaView style={dynamicStyles.container}>
      <StatusBar barStyle={isDark ? "light-content" : "dark-content"} backgroundColor={isDark ? "#1a1a1a" : "#ffffff"} />
      
      {/* Clean Perplexity-inspired Header */}
      <View style={dynamicStyles.header}>
        <TouchableOpacity 
          style={dynamicStyles.profileButton}
          onPress={() => navigation.navigate('Profile')}
          activeOpacity={0.7}
        >
          <View style={dynamicStyles.profileAvatar}>
            <Text style={dynamicStyles.profileAvatarText}>M</Text>
          </View>
        </TouchableOpacity>
        
        <Text style={dynamicStyles.appTitle}>Mews</Text>
        
        <TouchableOpacity 
          style={dynamicStyles.newsButton}
          onPress={() => navigation.navigate('Discover')}
          activeOpacity={0.7}
        >
          <Text style={dynamicStyles.newsButtonIcon}>📰</Text>
        </TouchableOpacity>
      </View>

      <View style={styles.content}>
        {/* Chat Screen - Main */}
        <Animated.View style={[
          styles.screen,
          {
            opacity: fadeAnim,
            transform: [{
              translateX: slideAnim.interpolate({
                inputRange: [0, 1],
                outputRange: [0, -screenWidth * 0.3],
              }),
            }],
          }
        ]}>
          <ChatScreen />
        </Animated.View>

        {/* News Discovery Screen - Slides in from right */}
        <Animated.View style={[
          styles.discoverScreen,
          {
            opacity: slideAnim,
            transform: [{
              translateX: slideAnim.interpolate({
                inputRange: [0, 1],
                outputRange: [screenWidth, 0],
              }),
            }],
          }
        ]}>
          <NewsDiscoveryScreen />
        </Animated.View>
      </View>
    </SafeAreaView>
  );
};

const createDynamicStyles = (isDark: boolean) => StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: isDark ? '#1a1a1a' : '#ffffff',
  },
  header: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 20,
    paddingVertical: 16,
    backgroundColor: isDark ? '#1a1a1a' : '#ffffff',
    borderBottomWidth: 1,
    borderBottomColor: isDark ? '#333' : '#f0f0f0',
  },
  profileButton: {
    width: 40,
    height: 40,
  },
  profileAvatar: {
    width: 40,
    height: 40,
    borderRadius: 20,
    backgroundColor: '#0a57d4',
    alignItems: 'center',
    justifyContent: 'center',
  },
  profileAvatarText: {
    color: '#ffffff',
    fontSize: 18,
    fontWeight: '600',
  },
  appTitle: {
    fontSize: 28,
    fontWeight: '700',
    color: isDark ? '#ffffff' : '#1a1a1a',
    letterSpacing: -0.5,
  },
  newsButton: {
    width: 40,
    height: 40,
    borderRadius: 20,
    backgroundColor: isDark ? '#333' : '#f8f9fa',
    alignItems: 'center',
    justifyContent: 'center',
  },
  newsButtonIcon: {
    fontSize: 20,
  },
});

// Static styles that don't change with theme
const styles = StyleSheet.create({
  content: {
    flex: 1,
    position: 'relative',
  },
  screen: {
    flex: 1,
    backgroundColor: '#ffffff',
  },
  screenShifted: {
    transform: [{ translateX: -screenWidth * 0.3 }],
  },
  discoverScreen: {
    position: 'absolute',
    top: 0,
    right: 0,
    width: screenWidth,
    height: '100%',
    backgroundColor: '#ffffff',
    shadowColor: '#000',
    shadowOffset: { width: -2, height: 0 },
    shadowOpacity: 0.1,
    shadowRadius: 8,
    elevation: 8,
  },
  discoverScreenVisible: {
    transform: [{ translateX: -screenWidth }],
  },
});

export default MainScreen;
