import React, { useState, useEffect } from 'react';
import {
  View,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  Text,
  SafeAreaView,
  StatusBar,
  Image,
  TextInput,
  Alert,
  ActivityIndicator,
  Linking,
} from 'react-native';
import { useNavigation, useRoute } from '@react-navigation/native';
import { useTheme } from '../contexts/ThemeContext';
import { NewsArticle } from '../types';
import { apiService } from '../services/api';
import RenderHtml from 'react-native-render-html';

interface DeepDiveScreenProps {
  route: {
    params: {
      articleId: string;
    };
  };
}

const DeepDiveScreen: React.FC<DeepDiveScreenProps> = ({ route }) => {
  const navigation = useNavigation();
  const { isDark } = useTheme();
  const { articleId } = route.params;
  const [article, setArticle] = useState<NewsArticle | null>(null);
  const [loading, setLoading] = useState(true);
  const [question, setQuestion] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [aiResponse, setAiResponse] = useState('');

  useEffect(() => {
    fetchArticle();
  }, [articleId]);

  const fetchArticle = async () => {
    try {
      setLoading(true);
      const response = await apiService.getArticle(articleId);
      if (response.success) {
        setArticle(response.data);
      } else {
        Alert.alert('Error', 'Failed to load article');
        navigation.goBack();
      }
    } catch (error) {
      console.error('Error fetching article:', error);
      Alert.alert('Error', 'Failed to load article');
      navigation.goBack();
    } finally {
      setLoading(false);
    }
  };

  const handleAskQuestion = async () => {
    if (!question.trim() || !article) return;

    setIsLoading(true);
    try {
      const response = await apiService.askQuestion(articleId, question);
      if (response.success) {
        setAiResponse(response.response);
      } else {
        Alert.alert('Error', 'Failed to get AI response');
      }
    } catch (error) {
      console.error('Error asking question:', error);
      Alert.alert('Error', 'Failed to get AI response');
    } finally {
      setIsLoading(false);
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
    });
  };

  const handleLinkPress = (url: string) => {
    // Open links in external browser
    Linking.openURL(url).catch(err => {
      Alert.alert('Error', 'Could not open link');
    });
  };

  const dynamicStyles = createDynamicStyles(isDark);

  if (loading) {
    return (
      <SafeAreaView style={dynamicStyles.container}>
        <StatusBar barStyle={isDark ? "light-content" : "dark-content"} backgroundColor={isDark ? "#1a1a1a" : "#ffffff"} />
        <View style={dynamicStyles.loadingContainer}>
          <ActivityIndicator size="large" color="#0a57d4" />
          <Text style={dynamicStyles.loadingText}>Loading article...</Text>
        </View>
      </SafeAreaView>
    );
  }

  if (!article) {
    return (
      <SafeAreaView style={dynamicStyles.container}>
        <StatusBar barStyle={isDark ? "light-content" : "dark-content"} backgroundColor={isDark ? "#1a1a1a" : "#ffffff"} />
        <View style={dynamicStyles.errorContainer}>
          <Text style={dynamicStyles.errorText}>Failed to load article</Text>
          <TouchableOpacity style={dynamicStyles.retryButton} onPress={() => navigation.goBack()}>
            <Text style={dynamicStyles.retryButtonText}>Go Back</Text>
          </TouchableOpacity>
        </View>
      </SafeAreaView>
    );
  }

  return (
    <SafeAreaView style={dynamicStyles.container}>
      <StatusBar barStyle={isDark ? "light-content" : "dark-content"} backgroundColor={isDark ? "#1a1a1a" : "#ffffff"} />
      
      {/* Header */}
      <View style={dynamicStyles.header}>
        <TouchableOpacity
          style={dynamicStyles.backButton}
          onPress={() => navigation.goBack()}
          activeOpacity={0.7}
        >
          <Text style={dynamicStyles.backIcon}>←</Text>
        </TouchableOpacity>
        <Text style={dynamicStyles.headerTitle}>Deep Dive</Text>
        <TouchableOpacity
          style={dynamicStyles.shareButton}
          onPress={() => Alert.alert('Share', 'Share functionality coming soon')}
          activeOpacity={0.7}
        >
          <Text style={dynamicStyles.shareIcon}>📤</Text>
        </TouchableOpacity>
      </View>

      <ScrollView style={dynamicStyles.content} showsVerticalScrollIndicator={false}>
        {/* Article Header */}
        <View style={dynamicStyles.articleHeader}>
          {article.image && (
            <Image
              source={{ uri: article.image.startsWith('http') 
                ? article.image 
                : `http://192.168.100.227/mews/public${article.image}` }}
              style={dynamicStyles.articleImage}
              resizeMode="cover"
            />
          )}
          
          <View style={dynamicStyles.articleMeta}>
            <View style={dynamicStyles.categoryBadge}>
              <Text style={dynamicStyles.categoryText}>{article.category}</Text>
            </View>
            <Text style={dynamicStyles.articleDate}>{article.date}</Text>
          </View>
          
          <Text style={dynamicStyles.articleTitle}>{article.title}</Text>
          
          {/* <View style={dynamicStyles.articleStats}>
            <View style={dynamicStyles.statItem}>
              <Text style={dynamicStyles.statIcon}>👁️</Text>
              <Text style={dynamicStyles.statText}>{article.views || 0} views</Text>
            </View>
            <View style={dynamicStyles.statItem}>
              <Text style={dynamicStyles.statIcon}>📰</Text>
              <Text style={dynamicStyles.statText}>{article.source}</Text>
            </View>
          </View> */}
        </View>

        {/* Article Content */}
        <View style={dynamicStyles.articleContent}>
          <RenderHtml
            contentWidth={300}
            source={{ html: article.content || '' }}
            tagsStyles={{
              p: {
                fontSize: 16,
                lineHeight: 26,
                color: isDark ? '#e6e6e6' : '#333',
                marginBottom: 16,
                textAlign: 'justify',
              },
              h1: {
                fontSize: 24,
                fontWeight: '600',
                color: isDark ? '#ffffff' : '#1a1a1a',
                marginBottom: 16,
              },
              h2: {
                fontSize: 22,
                fontWeight: '600',
                color: isDark ? '#ffffff' : '#1a1a1a',
                marginBottom: 14,
              },
              h3: {
                fontSize: 20,
                fontWeight: '600',
                color: isDark ? '#ffffff' : '#1a1a1a',
                marginBottom: 12,
              },
              h4: {
                fontSize: 18,
                fontWeight: '600',
                color: isDark ? '#ffffff' : '#1a1a1a',
                marginBottom: 10,
              },
              h5: {
                fontSize: 16,
                fontWeight: '600',
                color: isDark ? '#ffffff' : '#1a1a1a',
                marginBottom: 8,
              },
              h6: {
                fontSize: 14,
                fontWeight: '600',
                color: isDark ? '#ffffff' : '#1a1a1a',
                marginBottom: 6,
              },
              strong: {
                fontWeight: '600',
                color: isDark ? '#ffffff' : '#1a1a1a',
              },
              b: {
                fontWeight: '600',
                color: isDark ? '#ffffff' : '#1a1a1a',
              },
              em: {
                fontStyle: 'italic',
                color: isDark ? '#e6e6e6' : '#333',
              },
              i: {
                fontStyle: 'italic',
                color: isDark ? '#e6e6e6' : '#333',
              },
              ul: {
                marginBottom: 16,
                paddingLeft: 20,
              },
              ol: {
                marginBottom: 16,
                paddingLeft: 20,
              },
              li: {
                marginBottom: 8,
                color: isDark ? '#e6e6e6' : '#333',
              },
              blockquote: {
                marginBottom: 16,
                paddingLeft: 16,
                borderLeftWidth: 4,
                borderLeftColor: '#0a57d4',
                backgroundColor: isDark ? '#2d2d2d' : '#f8f9fa',
                padding: 16,
                borderRadius: 4,
              },
              img: {
                maxWidth: '100%',
                height: 'auto',
                borderRadius: 8,
                marginBottom: 16,
                marginTop: 16,
              },
              a: {
                color: '#0a57d4',
                textDecorationLine: 'none',
                fontWeight: '500',
              },
            }}
            onLinkPress={(event, href) => {
              handleLinkPress(href);
            }}
            renderers={{
              img: ({ src, alt, ...props }) => (
                <Image
                  source={{ uri: src }}
                  style={dynamicStyles.articleImage}
                  resizeMode="cover"
                  {...props}
                />
              ),
            }}
          />
        </View>

        {/* AI Q&A Section */}
        <View style={dynamicStyles.aiSection}>
          <View style={dynamicStyles.aiHeader}>
            <View style={dynamicStyles.aiAvatar}>
              <Text style={dynamicStyles.aiAvatarText}>M</Text>
            </View>
            <View style={dynamicStyles.aiTitleContainer}>
              <Text style={dynamicStyles.aiTitle}>Ask Mews AI</Text>
              <Text style={dynamicStyles.aiSubtitle}>Get insights about this article</Text>
            </View>
          </View>

          {/* Question Input */}
          <View style={dynamicStyles.questionContainer}>
            <TextInput
              style={dynamicStyles.questionInput}
              value={question}
              onChangeText={setQuestion}
              placeholder="Ask a question about this article..."
              placeholderTextColor={isDark ? "#8e8e93" : "#8e8e93"}
              multiline
              maxLength={500}
            />
            <TouchableOpacity
              style={[
                dynamicStyles.askButton,
                (!question.trim() || isLoading) && dynamicStyles.askButtonDisabled
              ]}
              onPress={handleAskQuestion}
              disabled={!question.trim() || isLoading}
              activeOpacity={0.7}
            >
              <Text style={dynamicStyles.askButtonText}>
                {isLoading ? '...' : 'Ask'}
              </Text>
            </TouchableOpacity>
          </View>

          {/* AI Response */}
          {aiResponse && (
            <View style={dynamicStyles.responseContainer}>
              <View style={dynamicStyles.responseHeader}>
                <View style={dynamicStyles.aiAvatar}>
                  <Text style={dynamicStyles.aiAvatarText}>M</Text>
                </View>
                <Text style={dynamicStyles.responseTitle}>Mews AI Response</Text>
              </View>
              <Text style={dynamicStyles.responseText}>{aiResponse}</Text>
            </View>
          )}

        </View>
      </ScrollView>

    </SafeAreaView>
  );
};


const createDynamicStyles = (isDark: boolean) => StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: isDark ? '#1a1a1a' : '#ffffff',
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    paddingVertical: 16,
    borderBottomWidth: 1,
    borderBottomColor: isDark ? '#404040' : '#f0f0f0',
  },
  backButton: {
    width: 40,
    height: 40,
    alignItems: 'center',
    justifyContent: 'center',
  },
  backIcon: {
    fontSize: 20,
    color: isDark ? '#ffffff' : '#1a1a1a',
  },
  headerTitle: {
    fontSize: 20,
    fontWeight: '600',
    color: isDark ? '#ffffff' : '#1a1a1a',
  },
  shareButton: {
    width: 40,
    height: 40,
    alignItems: 'center',
    justifyContent: 'center',
  },
  shareIcon: {
    fontSize: 20,
  },
  content: {
    flex: 1,
  },
  articleHeader: {
    padding: 20,
  },
  articleImage: {
    width: '100%',
    height: 200,
    borderRadius: 12,
    marginBottom: 16,
  },
  articleMeta: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: 16,
  },
  categoryBadge: {
    backgroundColor: '#0a57d4',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 12,
  },
  categoryText: {
    color: '#ffffff',
    fontSize: 12,
    fontWeight: '600',
  },
  articleDate: {
    fontSize: 14,
    color: isDark ? '#b3b3b3' : '#666',
  },
  articleTitle: {
    fontSize: 24,
    fontWeight: '700',
    color: isDark ? '#ffffff' : '#1a1a1a',
    lineHeight: 32,
    marginBottom: 16,
  },
  articleStats: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 20,
  },
  statItem: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 6,
  },
  statIcon: {
    fontSize: 14,
  },
  statText: {
    fontSize: 14,
    color: isDark ? '#b3b3b3' : '#666',
  },
  articleContent: {
    paddingHorizontal: 20,
    paddingBottom: 20,
    borderBottomWidth: 1,
    borderBottomColor: isDark ? '#404040' : '#f0f0f0',
  },
  contentText: {
    fontSize: 16,
    lineHeight: 26,
    color: isDark ? '#e6e6e6' : '#333',
    textAlign: 'justify',
  },
  articleImage: {
    width: '100%',
    height: 200,
    borderRadius: 8,
    marginVertical: 16,
  },
  aiSection: {
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: isDark ? '#404040' : '#f0f0f0',
  },
  aiHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 20,
  },
  aiAvatar: {
    width: 48,
    height: 48,
    borderRadius: 24,
    backgroundColor: '#0a57d4',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 16,
  },
  aiAvatarText: {
    color: '#ffffff',
    fontSize: 20,
    fontWeight: '600',
  },
  aiTitleContainer: {
    flex: 1,
  },
  aiTitle: {
    fontSize: 20,
    fontWeight: '600',
    color: isDark ? '#ffffff' : '#1a1a1a',
    marginBottom: 4,
  },
  aiSubtitle: {
    fontSize: 14,
    color: isDark ? '#b3b3b3' : '#666',
  },
  questionContainer: {
    flexDirection: 'row',
    alignItems: 'flex-end',
    backgroundColor: isDark ? '#2d2d2d' : '#f8f9fa',
    borderRadius: 16,
    padding: 16,
    marginBottom: 20,
    borderWidth: 1,
    borderColor: isDark ? '#555' : '#e9ecef',
  },
  questionInput: {
    flex: 1,
    fontSize: 16,
    color: isDark ? '#ffffff' : '#1a1a1a',
    maxHeight: 100,
    marginRight: 12,
  },
  askButton: {
    backgroundColor: '#0a57d4',
    paddingHorizontal: 20,
    paddingVertical: 12,
    borderRadius: 12,
  },
  askButtonDisabled: {
    backgroundColor: '#ccc',
  },
  askButtonText: {
    color: '#ffffff',
    fontSize: 16,
    fontWeight: '600',
  },
  responseContainer: {
    backgroundColor: isDark ? '#2d2d2d' : '#f8f9fa',
    borderRadius: 16,
    padding: 16,
    marginBottom: 20,
    borderWidth: 1,
    borderColor: isDark ? '#555' : '#e9ecef',
  },
  responseHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 12,
  },
  responseTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: isDark ? '#ffffff' : '#1a1a1a',
    marginLeft: 12,
  },
  responseText: {
    fontSize: 15,
    lineHeight: 22,
    color: isDark ? '#e6e6e6' : '#333',
  },
  loadingContainer: {
    flex: 1,
    alignItems: 'center',
    justifyContent: 'center',
    padding: 20,
  },
  loadingText: {
    fontSize: 16,
    color: isDark ? '#ffffff' : '#1a1a1a',
    marginTop: 16,
  },
  errorContainer: {
    flex: 1,
    alignItems: 'center',
    justifyContent: 'center',
    padding: 20,
  },
  errorText: {
    fontSize: 16,
    color: isDark ? '#ffffff' : '#1a1a1a',
    marginBottom: 20,
    textAlign: 'center',
  },
  retryButton: {
    backgroundColor: '#0a57d4',
    paddingHorizontal: 20,
    paddingVertical: 12,
    borderRadius: 12,
  },
  retryButtonText: {
    color: '#ffffff',
    fontSize: 16,
    fontWeight: '600',
  },
});

export default DeepDiveScreen;
