import React, { useState, useRef, useEffect } from 'react';
import {
  View,
  StyleSheet,
  ScrollView,
  KeyboardAvoidingView,
  Platform,
  Alert,
  TouchableOpacity,
  StatusBar,
  Image,
  TouchableWithoutFeedback,
  Keyboard,
  TextInput,
  Text,
  Dimensions,
} from 'react-native';
import { useDispatch, useSelector } from 'react-redux';
import { useNavigation } from '@react-navigation/native';
import { RootState, AppDispatch } from '../store';
import { sendChatMessage, clearMessages, addMessage } from '../store/slices/chatSlice';
import { ChatMessage, NewsArticle } from '../types';
import { useTheme } from '../contexts/ThemeContext';

const { width: screenWidth } = Dimensions.get('window');

const ChatScreen: React.FC = () => {
  const dispatch = useDispatch<AppDispatch>();
  const navigation = useNavigation();
  const { messages, isLoading } = useSelector((state: RootState) => state.chat);
  const { isDark } = useTheme();
  const [inputText, setInputText] = useState('');
  const scrollViewRef = useRef<ScrollView>(null);

  const categories = [
    { key: 'Politics', name: 'Politics' },
    { key: 'Business', name: 'Business' },
    { key: 'Sports', name: 'Sports' },
    { key: 'Entertainment', name: 'Entertainment' },
    { key: 'Technology', name: 'Technology' },
    { key: 'Health', name: 'Health' },
  ];

  useEffect(() => {
    if (messages.length > 0) {
      setTimeout(() => {
        scrollViewRef.current?.scrollToEnd({ animated: true });
      }, 100);
    }
  }, [messages]);

  // Placeholder for voice recognition (will be implemented in production build)
  const handleMicrophonePress = () => {
    Alert.alert(
      'Voice Recognition',
      'Voice recognition will be available in the production build. For now, please type your message.',
      [{ text: 'OK' }]
    );
  };

  const handleSendMessage = async () => {
    if (inputText.trim() === '') return;

    const query = inputText.trim();
    
    // Add user message immediately to show it while AI is thinking
    const userMessage: ChatMessage = {
      id: Date.now().toString(),
      text: query,
      isUser: true,
      timestamp: Date.now(),
    };
    
    // Dispatch user message immediately
    dispatch(addMessage(userMessage));
    
    setInputText('');

    try {
      await dispatch(sendChatMessage(query)).unwrap();
    } catch (error) {
      Alert.alert('Error', 'Failed to send message. Please try again.');
    }
  };

  const handleCategoryPress = async (category: string) => {
    const categoryQueries = {
      'Politics': 'What are the latest political news and developments?',
      'Business': 'What are the latest business and economic news?',
      'Sports': 'What are the latest sports news?',
      'Entertainment': 'What are the latest entertainment and celebrity news?',
      'Technology': 'What are the latest technology news?',
      'Health': 'What are the latest health and medical news?'
    };
    
    const query = categoryQueries[category as keyof typeof categoryQueries];
    if (query) {
      try {
        await dispatch(sendChatMessage(query)).unwrap();
      } catch (error) {
        Alert.alert('Error', 'Failed to send message. Please try again.');
      }
    }
  };

  const handleNewsPress = (article: NewsArticle) => {
    console.log('News card pressed:', article);
    (navigation as any).navigate('DeepDive', { articleId: article.id });
  };

  const handleDeepDive = (article: NewsArticle) => {
    console.log('Deep dive pressed:', article);
    (navigation as any).navigate('DeepDive', { articleId: article.id });
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString();
  };

  const formatTime = (timestamp: number) => {
    return new Date(timestamp).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
  };

  const dynamicStyles = createDynamicStyles(isDark);

  return (
    <TouchableWithoutFeedback onPress={Keyboard.dismiss}>
      <View style={dynamicStyles.container}>
        <KeyboardAvoidingView
          style={dynamicStyles.keyboardContainer}
          behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        >
          {/* Clean Landing Page */}
          {messages.length === 0 && (
            <View style={dynamicStyles.landingContainer}>
              <View style={dynamicStyles.logoContainer}>
                <Text style={dynamicStyles.logoText}>Mews</Text>
                <View style={dynamicStyles.proBadge}>
                  <Text style={dynamicStyles.proBadgeText}>AI</Text>
                </View>
              </View>
              <Text style={dynamicStyles.subtitle}>Ask anything about Ghana</Text>
            </View>
          )}

          {/* Messages */}
          <ScrollView 
            style={dynamicStyles.messagesContainer}
            contentContainerStyle={dynamicStyles.messagesContent}
            ref={scrollViewRef}
            onContentSizeChange={() => scrollViewRef.current?.scrollToEnd({ animated: true })}
            showsVerticalScrollIndicator={false}
          >
            {messages.map((message, index) => (
              <View key={index} style={[
                dynamicStyles.messageContainer,
                message.isUser ? dynamicStyles.userMessage : dynamicStyles.botMessage
              ]}>
                {!message.isUser && (
                  <View style={dynamicStyles.botAvatar}>
                    <Text style={dynamicStyles.botAvatarText}>M</Text>
                  </View>
                )}
                
                <View style={[
                  dynamicStyles.messageBubble,
                  message.isUser ? dynamicStyles.userBubble : dynamicStyles.botBubble
                ]}>
                  <Text style={[
                    dynamicStyles.messageText,
                    message.isUser ? dynamicStyles.userText : dynamicStyles.botText
                  ]}>
                    {message.text}
                  </Text>
                  
                  {/* Render news cards for bot messages - Web App Style */}
                  {!message.isUser && message.news_data && message.news_data.length > 0 && (
                    <View style={dynamicStyles.newsContainer}>
                      {message.news_data.map((article, articleIndex) => {
                        // Use transformed structure from API (matches web app)
                        const articleId = article.id;
                        const title = article.title;
                        const content = article.content;
                        const category = article.sub_category || article.category || 'General';
                        const source = article.source;
                        const views = article.views;
                        const date = article.date;
                        
                        // Fix image URL like web app
                        let imageUrl = 'https://via.placeholder.com/300x180?text=No+Image';
                        if (article.image && article.image !== 'N/A' && article.image.trim() !== '') {
                          imageUrl = article.image.startsWith('http') 
                            ? article.image 
                            : `http://192.168.100.227/mews/public${article.image}`;
                        }
                        
                        // Debug log
                        console.log('Rendering news card:', {
                          id: articleId,
                          title,
                          image: imageUrl,
                          category,
                          source,
                          date,
                          content: content.substring(0, 50) + '...'
                        });
                        
                        return (
                          <TouchableOpacity
                            key={articleIndex}
                            style={dynamicStyles.newsCard}
                            onPress={() => handleNewsPress(article)}
                            activeOpacity={0.8}
                          >
                            <View style={dynamicStyles.newsCardImageContainer}>
                              <Image 
                                source={{ uri: imageUrl }} 
                                style={dynamicStyles.newsCardImage}
                                resizeMode="cover"
                                onError={(error) => {
                                  console.log('Image load error:', error.nativeEvent.error);
                                }}
                              />
                              <View style={dynamicStyles.newsCardCategoryOverlay}>
                                <Text style={dynamicStyles.newsCardCategoryText}>{category}</Text>
                              </View>
                            </View>
                            
                            <View style={dynamicStyles.newsCardContentWrapper}>
                              <Text style={dynamicStyles.newsTitle} numberOfLines={2}>
                                {title}
                              </Text>
                              
                              <View style={dynamicStyles.newsCardContent}>
                                <Text style={dynamicStyles.newsCardExcerpt} numberOfLines={3}>
                                  {content}
                                </Text>
                              </View>
                              
                              <View style={dynamicStyles.newsMeta}>
                                <View style={dynamicStyles.newsMetaRow}>
                                  <Text style={dynamicStyles.newsDate}>
                                    📅 {date}
                                  </Text>
                                  <Text style={dynamicStyles.newsViews}>
                                    👁️ {views || 0}
                                  </Text>
                                </View>
                                <View style={dynamicStyles.newsSource}>
                                  <Text style={dynamicStyles.newsSourceText}>
                                    Deep Dive →
                                  </Text>
                                </View>
                              </View>
                            </View>
                          </TouchableOpacity>
                        );
                      })}
                      
                      {/* Tip for more news */}
                      <View style={dynamicStyles.newsTipContainer}>
                        <Text style={dynamicStyles.newsTipIcon}>💡</Text>
                        <Text style={dynamicStyles.newsTipText}>
                          <Text style={dynamicStyles.newsTipBold}>Tip:</Text> Type "more" to get additional news on this topic!
                        </Text>
                      </View>
                    </View>
                  )}
                  
                  <Text style={[
                    dynamicStyles.messageTime,
                    message.isUser ? dynamicStyles.userTime : dynamicStyles.botTime
                  ]}>
                    {formatTime(message.timestamp)}
                  </Text>
                </View>
              </View>
            ))}
            
            {isLoading && (
              <View style={dynamicStyles.loadingContainer}>
                <View style={dynamicStyles.botAvatar}>
                  <Text style={dynamicStyles.botAvatarText}>M</Text>
                </View>
                <View style={dynamicStyles.loadingBubble}>
                  <Text style={dynamicStyles.loadingText}>AI is thinking...</Text>
                </View>
              </View>
            )}
          </ScrollView>

          {/* Category Buttons - Above Input (Like Web App) */}
          <View style={dynamicStyles.categoriesContainer}>
            <ScrollView 
              horizontal 
              showsHorizontalScrollIndicator={false}
              contentContainerStyle={dynamicStyles.categoriesScroll}
            >
              {categories.map((category, index) => (
                <TouchableOpacity
                  key={index}
                  style={dynamicStyles.categoryButton}
                  onPress={() => handleCategoryPress(category.key)}
                  activeOpacity={0.7}
                >
                  <Text style={dynamicStyles.categoryText}>{category.name}</Text>
                </TouchableOpacity>
              ))}
            </ScrollView>
          </View>

          {/* Clean Input Area - Updated Design */}
          <View style={dynamicStyles.inputContainer}>
            <View style={dynamicStyles.inputWrapper}>
              <TextInput
                style={dynamicStyles.textInput}
                value={inputText}
                onChangeText={setInputText}
                placeholder="Ask anything..."
                placeholderTextColor={isDark ? "#8e8e93" : "#8e8e93"}
                multiline
                maxLength={500}
              />
              <TouchableOpacity
                style={dynamicStyles.microphoneButton}
                onPress={handleMicrophonePress}
                activeOpacity={0.7}
              >
                <Text style={dynamicStyles.microphoneIcon}>
                  🎤
                </Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[dynamicStyles.sendButton, inputText.trim() ? dynamicStyles.sendButtonActive : null]}
                onPress={handleSendMessage}
                disabled={!inputText.trim() || isLoading}
                activeOpacity={0.7}
              >
                <Text style={dynamicStyles.sendButtonIcon}>→</Text>
              </TouchableOpacity>
            </View>
          </View>
        </KeyboardAvoidingView>
      </View>
    </TouchableWithoutFeedback>
  );
};

const createDynamicStyles = (isDark: boolean) => StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: isDark ? '#1a1a1a' : '#ffffff',
  },
  keyboardContainer: {
    flex: 1,
  },
  landingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingHorizontal: 40,
  },
  logoContainer: {
    alignItems: 'center',
    marginBottom: 20,
  },
  logoText: {
    fontSize: 48,
    fontWeight: '700',
    color: isDark ? '#ffffff' : '#1a1a1a',
    letterSpacing: -2,
    marginBottom: 8,
  },
  proBadge: {
    backgroundColor: '#0a57d4',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 16,
    shadowColor: '#0a57d4',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
    elevation: 4,
  },
  proBadgeText: {
    color: '#ffffff',
    fontSize: 14,
    fontWeight: '600',
    letterSpacing: 1,
  },
  subtitle: {
    fontSize: 18,
    color: isDark ? '#b3b3b3' : '#666',
    fontWeight: '400',
    textAlign: 'center',
  },
  messagesContainer: {
    flex: 1,
    paddingHorizontal: 20,
  },
  messagesContent: {
    paddingVertical: 20,
  },
  messageContainer: {
    marginBottom: 20,
    flexDirection: 'row',
    alignItems: 'flex-start',
  },
  userMessage: {
    justifyContent: 'flex-end',
  },
  botMessage: {
    justifyContent: 'flex-start',
  },
  botAvatar: {
    width: 32,
    height: 32,
    borderRadius: 16,
    backgroundColor: '#0a57d4',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 12,
    marginTop: 4,
  },
  botAvatarText: {
    color: '#ffffff',
    fontSize: 14,
    fontWeight: '600',
  },
  messageBubble: {
    maxWidth: '80%',
    padding: 16,
    borderRadius: 20,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: isDark ? 0.3 : 0.1,
    shadowRadius: 8,
    elevation: 2,
  },
  userBubble: {
    backgroundColor: '#0a57d4',
    borderBottomRightRadius: 6,
    alignSelf: 'flex-end',
  },
  botBubble: {
    backgroundColor: isDark ? '#2d2d2d' : '#f8f9fa',
    borderBottomLeftRadius: 6,
    borderWidth: 1,
    borderColor: isDark ? '#404040' : '#e9ecef',
  },
  messageText: {
    fontSize: 16,
    lineHeight: 24,
    fontWeight: '400',
  },
  userText: {
    color: '#ffffff',
  },
  botText: {
    color: isDark ? '#e6e6e6' : '#1a1a1a',
  },
  messageTime: {
    fontSize: 12,
    marginTop: 8,
    opacity: 0.7,
  },
  userTime: {
    color: '#ffffff',
    textAlign: 'right',
  },
  botTime: {
    color: isDark ? '#b3b3b3' : '#666',
    textAlign: 'left',
  },
  newsContainer: {
    marginTop: 15,
    gap: 15,
  },
  newsCard: {
    backgroundColor: isDark ? '#2d2d2d' : '#ffffff',
    borderRadius: 12,
    overflow: 'hidden',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: isDark ? 0.3 : 0.1,
    shadowRadius: 8,
    elevation: 4,
    borderWidth: 1,
    borderColor: isDark ? '#404040' : '#f0f0f0',
    flexDirection: 'column',
  },
  newsCardImageContainer: {
    width: '100%',
    height: 180,
    position: 'relative',
    overflow: 'hidden',
  },
  newsCardImage: {
    width: '100%',
    height: '100%',
  },
  newsCardCategoryOverlay: {
    position: 'absolute',
    top: 10,
    left: 10,
    backgroundColor: '#0a57d4',
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 8,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 4,
    zIndex: 10,
  },
  newsCardCategoryText: {
    color: '#ffffff',
    fontSize: 10,
    fontWeight: '600',
  },
  newsCardContentWrapper: {
    padding: 12,
    flex: 1,
    flexDirection: 'column',
  },
  newsTitle: {
    color: isDark ? '#ffffff' : '#1a1a1a',
    marginBottom: 8,
    fontSize: 16,
    fontWeight: '600',
    lineHeight: 22,
  },
  newsCardContent: {
    marginBottom: 12,
  },
  newsCardExcerpt: {
    color: isDark ? '#b3b3b3' : '#666',
    fontSize: 13,
    lineHeight: 20,
  },
  newsMeta: {
    marginTop: 'auto',
    paddingTop: 10,
    borderTopWidth: 1,
    borderTopColor: isDark ? '#404040' : '#f0f0f0',
    flexDirection: 'column',
    gap: 8,
  },
  newsMetaRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  newsDate: {
    fontSize: 11,
    color: isDark ? '#b3b3b3' : '#666',
  },
  newsViews: {
    fontSize: 11,
    color: isDark ? '#b3b3b3' : '#666',
  },
  newsSource: {
    alignItems: 'flex-end',
  },
  newsSourceText: {
    fontSize: 11,
    color: '#0a57d4',
    fontWeight: '600',
  },
  newsTipContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: isDark ? '#404040' : '#f8f9fa',
    padding: 16,
    borderRadius: 12,
    marginTop: 12,
    borderWidth: 1,
    borderColor: isDark ? '#555' : '#e9ecef',
  },
  newsTipIcon: {
    fontSize: 16,
    marginRight: 12,
  },
  newsTipText: {
    fontSize: 13,
    color: isDark ? '#e6e6e6' : '#495057',
    flex: 1,
    fontWeight: '500',
  },
  newsTipBold: {
    fontWeight: '700',
    color: isDark ? '#ffffff' : '#1a1a1a',
  },
  loadingContainer: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    marginBottom: 20,
  },
  loadingBubble: {
    backgroundColor: isDark ? '#2d2d2d' : '#f8f9fa',
    padding: 16,
    borderRadius: 20,
    borderBottomLeftRadius: 6,
    borderWidth: 1,
    borderColor: isDark ? '#404040' : '#e9ecef',
    marginLeft: 12,
  },
  loadingText: {
    fontSize: 16,
    color: isDark ? '#b3b3b3' : '#666',
    fontStyle: 'italic',
  },
  inputContainer: {
    backgroundColor: isDark ? '#1a1a1a' : '#ffffff',
    paddingHorizontal: 20,
    paddingVertical: 16,
    borderTopWidth: 1,
    borderTopColor: isDark ? '#404040' : '#f0f0f0',
  },
  inputWrapper: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: isDark ? '#2d2d2d' : '#f8f9fa',
    borderRadius: 24,
    paddingHorizontal: 16,
    paddingVertical: 12,
    borderWidth: 1,
    borderColor: isDark ? '#555' : '#e9ecef',
  },
  textInput: {
    flex: 1,
    fontSize: 16,
    color: isDark ? '#ffffff' : '#1a1a1a',
    maxHeight: 100,
    paddingVertical: 4,
    marginRight: 8,
  },
  microphoneButton: {
    width: 36,
    height: 36,
    borderRadius: 18,
    backgroundColor: isDark ? '#404040' : '#ffffff',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 8,
    borderWidth: 1,
    borderColor: isDark ? '#555' : '#e9ecef',
  },
  microphoneIcon: {
    fontSize: 16,
  },
  sendButton: {
    backgroundColor: '#0a57d4',
    width: 36,
    height: 36,
    borderRadius: 18,
    alignItems: 'center',
    justifyContent: 'center',
  },
  sendButtonActive: {
    backgroundColor: '#0a57d4',
  },
  sendButtonIcon: {
    fontSize: 16,
    color: '#ffffff',
    fontWeight: '600',
  },
  categoriesContainer: {
    backgroundColor: isDark ? '#1a1a1a' : '#ffffff',
    paddingVertical: 16,
    borderTopWidth: 1,
    borderTopColor: isDark ? '#404040' : '#f0f0f0',
  },
  categoriesScroll: {
    paddingHorizontal: 20,
    gap: 12,
  },
  categoryButton: {
    backgroundColor: '#0a57d4',
    paddingHorizontal: 20,
    paddingVertical: 10,
    borderRadius: 20,
    alignItems: 'center',
    justifyContent: 'center',
    shadowColor: '#0a57d4',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
    elevation: 4,
  },
  categoryText: {
    color: '#ffffff',
    fontSize: 14,
    fontWeight: '600',
  },
});

export default ChatScreen;